<?php
/**
 * View current rules being used
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>📋 Current Rules Being Used</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 15px; border-radius: 5px; overflow-x: auto; }
    .rule-section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 5px; }
</style>';

global $wpdb;

try {
    $table_rulesets = $wpdb->prefix . 'coco_rulesets';
    
    // Get current active ruleset
    $ruleset = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    
    if (!$ruleset) {
        echo '<p class="error">❌ No active ruleset found!</p>';
        exit;
    }
    
    echo '<div class="rule-section">';
    echo '<h2>📊 Ruleset Information</h2>';
    echo '<p><strong>ID:</strong> ' . $ruleset->id . '</p>';
    echo '<p><strong>Version:</strong> ' . esc_html($ruleset->version) . '</p>';
    echo '<p><strong>Active:</strong> ' . ($ruleset->active ? 'Yes' : 'No') . '</p>';
    echo '<p><strong>Created:</strong> ' . $ruleset->created_at . '</p>';
    echo '</div>';
    
    echo '<div class="rule-section">';
    echo '<h2>📝 Raw YAML Content</h2>';
    echo '<pre>' . esc_html($ruleset->yaml_content) . '</pre>';
    echo '</div>';
    
    // Parse and display the rules in a readable format
    echo '<div class="rule-section">';
    echo '<h2>🎯 Parsed Rules (Human Readable)</h2>';
    
    $rules_engine = new Coco_Ops_Rules_Engine();
    
    // Test with different scenarios to show how rules work
    echo '<h3>📊 Example Calculations:</h3>';
    
    // Scenario 1: 200 people on Saturday
    echo '<h4>Scenario 1: 200 people on Saturday (Weekend)</h4>';
    $result1 = $rules_engine->compute_staffing(200, ['weekday' => 'Saturday', 'start_bucket' => 'evening']);
    echo '<ul>';
    foreach ($result1['staffing'] as $role => $data) {
        echo '<li><strong>' . ucwords(str_replace('_', ' ', $role)) . ':</strong> ' . $data['count'] . ' staff</li>';
    }
    echo '</ul>';
    
    // Scenario 2: 30 people on Tuesday (weekday substitution)
    echo '<h4>Scenario 2: 30 people on Tuesday (Weekday Substitution)</h4>';
    $result2 = $rules_engine->compute_staffing(30, ['weekday' => 'Tuesday', 'start_bucket' => 'evening']);
    echo '<ul>';
    foreach ($result2['staffing'] as $role => $data) {
        echo '<li><strong>' . ucwords(str_replace('_', ' ', $role)) . ':</strong> ' . $data['count'] . ' staff</li>';
    }
    echo '</ul>';
    
    // Scenario 3: 100 people on Friday
    echo '<h4>Scenario 3: 100 people on Friday</h4>';
    $result3 = $rules_engine->compute_staffing(100, ['weekday' => 'Friday', 'start_bucket' => 'evening']);
    echo '<ul>';
    foreach ($result3['staffing'] as $role => $data) {
        echo '<li><strong>' . ucwords(str_replace('_', ' ', $role)) . ':</strong> ' . $data['count'] . ' staff</li>';
    }
    echo '</ul>';
    
    echo '</div>';
    
    // Show detailed rationale for one scenario
    echo '<div class="rule-section">';
    echo '<h2>🔍 Detailed Calculation (200 people on Saturday)</h2>';
    echo '<pre>' . esc_html($result1['rationale_md']) . '</pre>';
    echo '</div>';
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<p>Stack trace:</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p><strong>💡 To make adjustments:</strong></p>';
echo '<ul>';
echo '<li>Edit the YAML content above</li>';
echo '<li>Or run the update script: <code>update-ruleset-simple.php</code></li>';
echo '<li>Or create a new ruleset with different ratios</li>';
echo '</ul>';
?>
