<?php
/**
 * Update existing ruleset with Cococure's specific ratios
 * Visit: /wp-content/plugins/coco-ops-forecast/update-ruleset.php
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔄 Update Ruleset with Cococure Ratios</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
</style>';

global $wpdb;

try {
    $table_rulesets = $wpdb->prefix . 'coco_rulesets';
    
    // Get current ruleset
    $current = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    
    if (!$current) {
        echo '<p class="error">❌ No active ruleset found!</p>';
        exit;
    }
    
    echo '<p class="info">📋 Current ruleset: Version ' . esc_html($current->version) . ' (ID: ' . $current->id . ')</p>';
    
    // New Cococure-specific YAML content
    $new_yaml_content = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022  # 1 security per 45 people
    weekend_multiplier: 1.15  # +15% on weekends
    weekday_min: 1  # Minimum 1 on weekdays
    weekend_min: 2  # Minimum 2 on weekends
    max_count: 20
    
  bartenders:
    base_ratio: 0.023  # 1 bartender per 43 people
    weekend_multiplier: 1.20  # +20% on weekends
    min_count: 2
    max_count: 4  # Limited by bar space
    
  wait_staff:
    base_ratio: 0.008  # 1 wait staff per 125 people
    weekend_multiplier: 1.50  # +50% on weekends
    min_count: 1
    max_count: 15
    
  bottle_girls:
    base_ratio: 0.010  # 1 bottle girl per 100 people
    weekend_multiplier: 1.15  # +15% on weekends
    min_count: 0
    max_count: 10
    
  managers:
    base_ratio: 0.002  # 1 manager per 500 people
    weekend_multiplier: 1.00  # Always 1 needed
    min_count: 1
    max_count: 1
    weekday_substitution: true  # Managers replace 1 bartender on weekdays when < 40 people";
    
    // Update the existing ruleset
    $result = $wpdb->update(
        $table_rulesets,
        [
            'yaml_content' => $new_yaml_content
        ],
        ['id' => $current->id],
        ['%s'],
        ['%d']
    );
    
    if ($result !== false) {
        echo '<p class="success">✅ Ruleset updated successfully!</p>';
        echo '<h2>New Cococure Staffing Ratios:</h2>';
        echo '<ul>';
        echo '<li><strong>Security:</strong> 1 per 45 people (2.2%) - Min 1 weekdays, Min 2 weekends</li>';
        echo '<li><strong>Bartenders:</strong> 1 per 43 people (2.3%) - MAX 4 (bar space limit)</li>';
        echo '<li><strong>Wait Staff:</strong> 1 per 125 people (0.8%) - +50% on weekends</li>';
        echo '<li><strong>Bottle Girls:</strong> 1 per 100 people (1.0%)</li>';
        echo '<li><strong>Managers:</strong> 1 per 500 people (0.2%) - Always 1 needed</li>';
        echo '</ul>';
        echo '<p><strong>Weekend Multipliers:</strong> +15% security, +20% bartenders, +50% wait staff, +15% bottle girls</p>';
        echo '<p><strong>Special Rules:</strong> Managers replace 1 bartender on weekdays when < 40 people expected</p>';
        
        echo '<h3>🧮 Example for 200 people on Saturday:</h3>';
        echo '<ul>';
        echo '<li><strong>Security:</strong> 200 × 0.022 × 1.15 = 5.06 → 5 staff</li>';
        echo '<li><strong>Bartenders:</strong> min(200 × 0.023 × 1.20, 4) = 4 staff (capped)</li>';
        echo '<li><strong>Wait Staff:</strong> 200 × 0.008 × 1.50 = 2.4 → 3 staff</li>';
        echo '<li><strong>Bottle Girls:</strong> 200 × 0.010 × 1.15 = 2.3 → 3 staff</li>';
        echo '<li><strong>Managers:</strong> 1 staff</li>';
        echo '</ul>';
        
    } else {
        echo '<p class="error">❌ Failed to update ruleset: ' . $wpdb->last_error . '</p>';
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
}

echo '<hr>';
echo '<p style="color: #666; font-size: 12px;">Delete this file after updating.</p>';
?>
