<?php
/**
 * Simple ruleset update - handles loading issues
 */

// Set time limit
set_time_limit(30);

// Add error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Try to load WordPress
$wp_load_paths = [
    '../../../wp-load.php',
    '../../../../wp-load.php', 
    '../../../../../wp-load.php'
];

$wp_loaded = false;
foreach ($wp_load_paths as $path) {
    if (file_exists($path)) {
        require_once($path);
        $wp_loaded = true;
        break;
    }
}

if (!$wp_loaded) {
    die('WordPress not found. Check file paths.');
}

// Check if we can access WordPress
if (!function_exists('current_user_can')) {
    die('WordPress not properly loaded.');
}

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔄 Update Ruleset (Simple Version)</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
</style>';

global $wpdb;

echo '<p>Starting update...</p>';

try {
    $table_rulesets = $wpdb->prefix . 'coco_rulesets';
    
    echo '<p>Checking table: ' . $table_rulesets . '</p>';
    
    // Check if table exists
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_rulesets'");
    if (!$table_exists) {
        echo '<p class="error">❌ Table does not exist: ' . $table_rulesets . '</p>';
        exit;
    }
    
    echo '<p class="info">✅ Table exists</p>';
    
    // Get current ruleset
    $current = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    
    if (!$current) {
        echo '<p class="error">❌ No active ruleset found!</p>';
        exit;
    }
    
    echo '<p class="info">📋 Found ruleset ID: ' . $current->id . '</p>';
    
    // Simple YAML content
    $new_yaml = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022
    weekend_multiplier: 1.15
    weekday_min: 1
    weekend_min: 2
    max_count: 20
  bartenders:
    base_ratio: 0.023
    weekend_multiplier: 1.20
    min_count: 2
    max_count: 4
  wait_staff:
    base_ratio: 0.008
    weekend_multiplier: 1.50
    min_count: 1
    max_count: 15
  bottle_girls:
    base_ratio: 0.010
    weekend_multiplier: 1.15
    min_count: 0
    max_count: 10
  managers:
    base_ratio: 0.002
    weekend_multiplier: 1.00
    min_count: 1
    max_count: 1
    weekday_substitution: true";
    
    echo '<p>Updating ruleset...</p>';
    
    // Update
    $result = $wpdb->update(
        $table_rulesets,
        ['yaml_content' => $new_yaml],
        ['id' => $current->id],
        ['%s'],
        ['%d']
    );
    
    if ($result !== false) {
        echo '<p class="success">✅ Ruleset updated successfully!</p>';
        echo '<p>New ratios applied:</p>';
        echo '<ul>';
        echo '<li>Security: 1 per 45 people (2.2%)</li>';
        echo '<li>Bartenders: 1 per 43 people (2.3%) - MAX 4</li>';
        echo '<li>Wait Staff: 1 per 125 people (0.8%) - +50% weekends</li>';
        echo '<li>Bottle Girls: 1 per 100 people (1.0%)</li>';
        echo '<li>Managers: 1 per 500 people (0.2%)</li>';
        echo '</ul>';
    } else {
        echo '<p class="error">❌ Update failed: ' . $wpdb->last_error . '</p>';
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
}

echo '<hr>';
echo '<p>Done.</p>';
?>
