<?php
/**
 * Test snapshot query to debug data collection
 * Visit: /wp-content/plugins/coco-ops-forecast/test-snapshot-query.php
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Snapshot Query Test</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    table { border-collapse: collapse; width: 100%; margin: 20px 0; }
    th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
    th { background: #4F46E5; color: white; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .warning { color: #D97706; }
</style>';

global $wpdb;

// Get an upcoming event
$event = $wpdb->get_row("
    SELECT p.ID, p.post_title, pm.meta_value as start_date
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_EventStartDate'
    WHERE p.post_type = 'tribe_events'
    AND p.post_status = 'publish'
    AND pm.meta_value > NOW()
    ORDER BY pm.meta_value ASC
    LIMIT 1
");

if (!$event) {
    echo '<p class="error">❌ No upcoming events found!</p>';
    exit;
}

echo '<h2>Testing Event</h2>';
echo '<p><strong>Event:</strong> ' . esc_html($event->post_title) . ' (ID: ' . $event->ID . ')</p>';
echo '<p><strong>Start Date:</strong> ' . $event->start_date . '</p>';

// Check if event has products
echo '<h2>1. Products Linked to Event</h2>';
$products = $wpdb->get_results($wpdb->prepare("
    SELECT 
        p.ID,
        p.post_title,
        pm_price.meta_value as price
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm_event ON p.ID = pm_event.post_id AND pm_event.meta_key = '_tribe_wooticket_for_event'
    LEFT JOIN {$wpdb->postmeta} pm_price ON p.ID = pm_price.post_id AND pm_price.meta_key = '_price'
    WHERE pm_event.meta_value = %d
    AND p.post_type = 'product'
", $event->ID));

if (empty($products)) {
    echo '<p class="error">❌ No products found for this event!</p>';
} else {
    echo '<p class="success">✅ Found ' . count($products) . ' products</p>';
    echo '<table>';
    echo '<tr><th>Product ID</th><th>Product Name</th><th>Price</th><th>Type</th></tr>';
    foreach ($products as $product) {
        $price = (float) $product->price;
        $type = $price >= 100 ? '<strong>TABLE</strong>' : 'ticket';
        echo '<tr>';
        echo '<td>' . $product->ID . '</td>';
        echo '<td>' . esc_html($product->post_title) . '</td>';
        echo '<td>£' . number_format($price, 2) . '</td>';
        echo '<td>' . $type . '</td>';
        echo '</tr>';
    }
    echo '</table>';
}

// Check if products have orders
echo '<h2>2. Orders for These Products</h2>';
$orders = $wpdb->get_results($wpdb->prepare("
    SELECT 
        wc_orders.id as order_id,
        wc_orders.status,
        p_ticket.post_title as ticket_name,
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    ORDER BY wc_orders.status, pm_price.meta_value DESC
", $event->ID));

if (empty($orders)) {
    echo '<p class="warning">⚠️ No orders found for this event (any status)</p>';
} else {
    echo '<p class="success">✅ Found ' . count($orders) . ' orders (all statuses)</p>';
    echo '<table>';
    echo '<tr><th>Order ID</th><th>Status</th><th>Ticket</th><th>Price</th><th>Qty</th><th>Total</th><th>Type</th></tr>';
    
    $total_tickets = 0;
    $total_tables = 0;
    $total_revenue = 0;
    
    foreach ($orders as $order) {
        $price = (float) $order->ticket_price;
        $qty = (int) $order->quantity;
        $line_total = (float) $order->line_total;
        $is_table = $price >= 100;
        $type = $is_table ? '<strong>TABLE</strong>' : 'ticket';
        
        $status_class = in_array($order->status, ['wc-completed', 'wc-processing']) ? 'success' : 'warning';
        
        if (in_array($order->status, ['wc-completed', 'wc-processing']) && $order->status != 'wc-refunded') {
            if ($is_table) {
                $total_tables += $qty;
            } else {
                $total_tickets += $qty;
            }
            $total_revenue += $line_total;
        }
        
        echo '<tr>';
        echo '<td>' . $order->order_id . '</td>';
        echo '<td class="' . $status_class . '">' . $order->status . '</td>';
        echo '<td>' . esc_html($order->ticket_name) . '</td>';
        echo '<td>£' . number_format($price, 2) . '</td>';
        echo '<td>' . $qty . '</td>';
        echo '<td>£' . number_format($line_total, 2) . '</td>';
        echo '<td>' . $type . '</td>';
        echo '</tr>';
    }
    echo '</table>';
    
    echo '<h3>Summary (completed/processing only):</h3>';
    echo '<ul>';
    echo '<li><strong>Tickets:</strong> ' . $total_tickets . '</li>';
    echo '<li><strong>Tables:</strong> ' . $total_tables . '</li>';
    echo '<li><strong>Revenue:</strong> £' . number_format($total_revenue, 2) . '</li>';
    echo '</ul>';
}

// Check existing snapshots
echo '<h2>3. Existing Snapshots for This Event</h2>';
$snapshots = $wpdb->get_results($wpdb->prepare("
    SELECT *
    FROM {$wpdb->prefix}coco_event_snapshots
    WHERE event_id = %d
    ORDER BY snapshot_ts DESC
    LIMIT 10
", $event->ID));

if (empty($snapshots)) {
    echo '<p class="warning">⚠️ No snapshots found yet. Click \"Trigger Snapshots Now\" on the Debug page.</p>';
} else {
    echo '<p class="success">✅ Found ' . count($snapshots) . ' snapshots</p>';
    echo '<table>';
    echo '<tr><th>Snapshot Time</th><th>Days to Event</th><th>Tickets</th><th>Tables</th><th>Revenue</th></tr>';
    foreach ($snapshots as $snapshot) {
        echo '<tr>';
        echo '<td>' . $snapshot->snapshot_ts . '</td>';
        echo '<td>' . number_format($snapshot->days_to_event, 1) . '</td>';
        echo '<td>' . $snapshot->tickets_sold . '</td>';
        echo '<td>' . $snapshot->tables_sold . '</td>';
        echo '<td>£' . number_format($snapshot->revenue_to_date, 2) . '</td>';
        echo '</tr>';
    }
    echo '</table>';
}

echo '<hr>';
echo '<p style="color: #666; font-size: 12px;">This is a diagnostic script. Delete after debugging.</p>';
?>
