<?php
/**
 * Simple test of save plan method
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🧪 Test Save Plan Method</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

try {
    // Get the event with planning data
    $table_planning = $wpdb->prefix . 'coco_event_planning';
    $planning_event = $wpdb->get_row("SELECT * FROM $table_planning WHERE est_attendance > 0 LIMIT 1");
    
    if (!$planning_event) {
        echo '<p class="error">❌ No planning data found</p>';
        exit;
    }
    
    $event_id = $planning_event->event_id;
    echo '<p>Testing with event ' . $event_id . '</p>';
    echo '<p>Planning data: ' . $planning_event->est_attendance . ' people, ' . $planning_event->est_tables . ' tables</p>';
    
    // Check if event features exist
    $table_features = $wpdb->prefix . 'coco_event_features';
    $features = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_features WHERE event_id = %d",
        $event_id
    ));
    
    if ($features) {
        echo '<p class="success">✅ Event features found</p>';
        echo '<p>Weekday: ' . $features->weekday . ', Start bucket: ' . $features->start_bucket . '</p>';
    } else {
        echo '<p class="error">❌ No event features found for event ' . $event_id . '</p>';
        exit;
    }
    
    // Test rules engine directly
    echo '<h2>🧮 Testing Rules Engine</h2>';
    $rules_engine = new Coco_Ops_Rules_Engine();
    $result = $rules_engine->compute_staffing(
        $planning_event->est_attendance,
        [
            'weekday' => $features->weekday,
            'start_bucket' => $features->start_bucket
        ],
        $planning_event->est_tables ?? 0
    );
    
    if (isset($result['staffing'])) {
        echo '<p class="success">✅ Rules engine working</p>';
        echo '<h3>Calculated Staffing:</h3>';
        echo '<ul>';
        foreach ($result['staffing'] as $role => $data) {
            echo '<li><strong>' . ucwords(str_replace('_', ' ', $role)) . ':</strong> ' . $data['count'] . '</li>';
        }
        echo '</ul>';
    } else {
        echo '<p class="error">❌ Rules engine failed</p>';
        echo '<pre>' . print_r($result, true) . '</pre>';
        exit;
    }
    
    // Test the save_plan method
    echo '<h2>💾 Testing Save Plan Method</h2>';
    
    // Create a mock request object
    class MockRequest {
        private $params;
        
        public function __construct($params) {
            $this->params = $params;
        }
        
        public function get_param($param) {
            return $this->params[$param] ?? null;
        }
    }
    
    $mock_request = new MockRequest([
        'id' => $event_id,
        'est_attendance' => $planning_event->est_attendance,
        'est_tables' => $planning_event->est_tables,
        'notes' => 'Test notes',
        'use_forecast' => false
    ]);
    
    // Call the save_plan method
    $rest_api = new Coco_Ops_REST_API();
    $response = $rest_api->save_plan($mock_request);
    
    if ($response && $response->data['success']) {
        echo '<p class="success">✅ Save plan method working</p>';
        
        // Check if staffing was saved
        $table_staffing = $wpdb->prefix . 'coco_staffing_recs';
        $saved_staffing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_staffing WHERE event_id = %d ORDER BY created_at DESC LIMIT 1",
            $event_id
        ));
        
        if ($saved_staffing) {
            echo '<p class="success">✅ Staffing recommendation saved!</p>';
            echo '<h3>Saved Staffing:</h3>';
            echo '<ul>';
            echo '<li><strong>Security:</strong> ' . $saved_staffing->rec_security . '</li>';
            echo '<li><strong>Bartenders:</strong> ' . $saved_staffing->rec_bartenders . '</li>';
            echo '<li><strong>Wait Staff:</strong> ' . $saved_staffing->rec_wait . '</li>';
            echo '<li><strong>Bottle Girls:</strong> ' . $saved_staffing->rec_bottle_girls . '</li>';
            echo '<li><strong>Managers:</strong> ' . $saved_staffing->rec_managers . '</li>';
            echo '</ul>';
        } else {
            echo '<p class="error">❌ Staffing recommendation not saved</p>';
        }
    } else {
        echo '<p class="error">❌ Save plan method failed</p>';
        if ($response) {
            echo '<pre>' . print_r($response, true) . '</pre>';
        }
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p>Test complete.</p>';
?>
