<?php
// Test the rule enforcement directly
require_once('../../../wp-config.php');

$event_id = 250474;

echo "🧪 Testing Rule Enforcement for Event $event_id\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// Get the sales data using the same method as the forecasting engine
$sales_data = $wpdb->get_results($wpdb->prepare("
    SELECT 
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event_id));

$paid_tickets = 0;
$free_tickets = 0;
$table_entries = 0;
$table_threshold = 100;

echo "📊 Sales Analysis:\n";
echo "Found " . count($sales_data) . " sales records\n\n";

foreach ($sales_data as $sale) {
    $price = (float) $sale->ticket_price;
    $qty = (int) $sale->quantity;
    $line_total = (float) $sale->line_total;
    
    if ($price >= $table_threshold) {
        $table_entries += $qty * 6;
    } elseif ($line_total > 0) {
        $paid_tickets += $qty;
    } else {
        $free_tickets += $qty;
    }
}

$minimum_forecast = $paid_tickets + $table_entries;

echo "Summary:\n";
echo "Paid tickets: $paid_tickets\n";
echo "Free tickets: $free_tickets\n";
echo "Table entries: $table_entries\n";
echo "Minimum forecast: $minimum_forecast\n\n";

// Test the max() function directly
$original_forecast = 115;
$corrected_forecast = max($original_forecast, $minimum_forecast);

echo "🧮 Rule Enforcement Test:\n";
echo "Original forecast: $original_forecast\n";
echo "Minimum required: $minimum_forecast\n";
echo "max($original_forecast, $minimum_forecast) = $corrected_forecast\n";

if ($corrected_forecast >= $minimum_forecast) {
    echo "✅ Rule enforcement working correctly\n";
} else {
    echo "❌ Rule enforcement failed\n";
}

// Test the actual forecasting engine
echo "\n🤖 Testing Actual Forecasting Engine:\n";
try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    $forecast = $forecasting_engine->generate_forecast($event_id);
    
    if ($forecast && isset($forecast['forecast'])) {
        $forecast_tickets = $forecast['forecast']['ticket_sales_p50'] ?? 0;
        echo "Generated forecast p50: $forecast_tickets\n";
        echo "Expected minimum: $minimum_forecast\n";
        
        if ($forecast_tickets >= $minimum_forecast) {
            echo "✅ SUCCESS: Forecast respects the rule\n";
        } else {
            echo "❌ FAILED: Forecast does not respect the rule\n";
            echo "This suggests the rule enforcement is not working in the forecasting engine\n";
        }
    } else {
        echo "❌ Failed to generate forecast\n";
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Test complete.\n";
?>
