<?php
/**
 * Test the current-sales endpoint directly
 * Visit: /wp-content/plugins/coco-ops-forecast/test-current-sales.php
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Test Current Sales Endpoint</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    pre { background: #f5f5f5; padding: 15px; border-radius: 5px; overflow-x: auto; }
    .success { color: #059669; }
    .error { color: #DC2626; }
</style>';

// Get an upcoming event
global $wpdb;
$event = $wpdb->get_row("
    SELECT p.ID, p.post_title, pm.meta_value as start_date
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_EventStartDate'
    WHERE p.post_type = 'tribe_events'
    AND p.post_status = 'publish'
    AND pm.meta_value > NOW()
    ORDER BY pm.meta_value ASC
    LIMIT 1
");

if (!$event) {
    echo '<p class="error">❌ No upcoming events found!</p>';
    exit;
}

echo '<h2>Testing Event</h2>';
echo '<p><strong>Event:</strong> ' . esc_html($event->post_title) . ' (ID: ' . $event->ID . ')</p>';
echo '<p><strong>Start Date:</strong> ' . $event->start_date . '</p>';

// Test the endpoint directly
echo '<h2>Testing /events/{id}/current-sales endpoint</h2>';

// Simulate the REST API call
$request = new WP_REST_Request('GET', '/coco-ops/v1/events/' . $event->ID . '/current-sales');
$request->set_param('id', $event->ID);

// Get the REST API handler
$rest_api = new Coco_Ops_REST_API();
$response = $rest_api->get_current_sales($request);

echo '<h3>Response:</h3>';
echo '<pre>' . json_encode($response->get_data(), JSON_PRETTY_PRINT) . '</pre>';

// Also test the direct query
echo '<h2>Direct Database Query Test</h2>';

$settings = get_option('coco_ops_settings', ['table_price_threshold' => 100.00]);
$table_threshold = isset($settings['table_price_threshold']) ? (float) $settings['table_price_threshold'] : 100.00;

$orders = $wpdb->get_results($wpdb->prepare("
    SELECT 
        p_ticket.post_title as ticket_name,
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event->ID));

echo '<h3>Raw Query Results:</h3>';
echo '<pre>' . json_encode($orders, JSON_PRETTY_PRINT) . '</pre>';

$tickets = 0;
$tables = 0;
$revenue = 0;

foreach ($orders as $order) {
    $price = (float) $order->ticket_price;
    $qty = (int) $order->quantity;
    $line_total = (float) $order->line_total;
    
    if ($price >= $table_threshold) {
        $tables += $qty;
    } else {
        $tickets += $qty;
    }
    
    $revenue += $line_total;
}

echo '<h3>Calculated Totals:</h3>';
echo '<ul>';
echo '<li><strong>Tickets:</strong> ' . $tickets . '</li>';
echo '<li><strong>Tables:</strong> ' . $tables . '</li>';
echo '<li><strong>Revenue:</strong> £' . number_format($revenue, 2) . '</li>';
echo '<li><strong>Table Threshold:</strong> £' . $table_threshold . '</li>';
echo '</ul>';

echo '<hr>';
echo '<p style="color: #666; font-size: 12px;">This is a diagnostic script. Delete after debugging.</p>';
?>
