<?php
/**
 * Test Cohort Matcher Directly
 */

// Include WordPress
require_once('../../../wp-config.php');

global $wpdb;

echo "🧪 Testing Cohort Matcher Directly\n";
echo "Time: " . current_time('mysql') . "\n\n";

// Test with the current event
$event_id = 256352; // TWNTY7 - 29 Nights

echo "📅 Testing event ID: {$event_id}\n";

// Get event features
$features = $wpdb->get_row($wpdb->prepare(
    "SELECT * FROM {$wpdb->prefix}coco_event_features WHERE event_id = %d",
    $event_id
));

if (!$features) {
    echo "❌ No event features found\n";
    exit;
}

echo "✅ Event features:\n";
echo "  - Venue: {$features->venue_id}\n";
echo "  - Weekday: {$features->weekday}\n";
echo "  - Start bucket: {$features->start_bucket}\n";
echo "  - Start time: {$features->start_ts}\n\n";

// Test the cohort matcher directly
echo "🔍 Testing cohort matcher directly...\n";

try {
    $cohort_matcher = new Coco_Ops_Cohort_Matcher();
    
    // Test with loose sensitivity (should find the most matches)
    echo "\n📊 Testing loose sensitivity:\n";
    $cohorts = $cohort_matcher->find_cohorts($event_id, 5, 'loose');
    
    echo "  - Found " . count($cohorts) . " cohorts\n";
    
    if (!empty($cohorts)) {
        echo "  - Top cohorts:\n";
        foreach (array_slice($cohorts, 0, 5) as $cohort) {
            echo "    * Event {$cohort['event_id']}: {$cohort['similarity_score']}% similar\n";
        }
    } else {
        echo "  - No cohorts found\n";
    }
    
    // Let's also check what the query is actually doing
    echo "\n🔍 Debugging the cohort query...\n";
    
    // Manually run the query to see what it's finding
    $table_features = $wpdb->prefix . 'coco_event_features';
    $table_actuals = $wpdb->prefix . 'coco_event_actuals';
    $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
    
    $query = "
        SELECT 
            ef.event_id,
            ef.venue_id,
            ef.weekday,
            ef.start_bucket,
            ef.start_ts,
            COALESCE(ea.final_attendance, 0) as final_attendance,
            COALESCE(ea.final_revenue, 0) as final_revenue
        FROM $table_features ef
        LEFT JOIN $table_actuals ea ON ef.event_id = ea.event_id
        LEFT JOIN $table_snapshots es ON ef.event_id = es.event_id
        WHERE ef.event_id != %d
        AND ef.venue_id = %s
        AND ef.holiday_flag = %d
        AND (ea.event_id IS NOT NULL OR es.event_id IS NOT NULL)
        AND ef.start_ts < NOW()
        ORDER BY ef.start_ts DESC
        LIMIT 10
    ";
    
    $results = $wpdb->get_results($wpdb->prepare($query, $event_id, $features->venue_id, $features->holiday_flag));
    
    echo "  - Query found " . count($results) . " events\n";
    foreach ($results as $result) {
        echo "    * Event {$result->event_id}: {$result->weekday} {$result->start_bucket} ({$result->start_ts})\n";
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

echo "\n🏁 Cohort matcher test complete\n";
?>
