<div class="wrap" x-data="reportsDashboard()">
    <h1 class="wp-heading-inline">📊 Performance Reports</h1>
    <hr class="wp-header-end">
    
    <!-- Filters -->
    <div class="coco-card" style="margin-bottom: 20px;">
        <h2 class="coco-card-title">Filters</h2>
        <div class="coco-filters">
            <div class="coco-filter-item">
                <label class="coco-label">Venue</label>
                <select x-model="filters.venue_id" @change="loadReports" class="coco-select">
                    <option value="">All Venues</option>
                    <template x-for="venue in venues" :key="venue.id">
                        <option :value="venue.id" x-text="venue.name"></option>
                    </template>
                </select>
            </div>
            
            <div class="coco-filter-item">
                <label class="coco-label">Date Range</label>
                <input type="date" x-model="filters.start_date" @change="loadReports" class="coco-input">
                <input type="date" x-model="filters.end_date" @change="loadReports" class="coco-input">
            </div>
            
            <div class="coco-filter-item">
                <label class="coco-label">Metric</label>
                <select x-model="filters.metric" @change="loadReports" class="coco-select">
                    <option value="revenue_per_staff">Revenue per Staff</option>
                    <option value="attendance_accuracy">Attendance Accuracy</option>
                    <option value="staff_efficiency">Staff Efficiency</option>
                    <option value="revenue_variance">Revenue Variance</option>
                </select>
            </div>
        </div>
    </div>
    
    <!-- Summary Cards -->
    <div class="coco-stats-grid" style="margin-bottom: 20px;">
        <div class="coco-stat-card">
            <div class="coco-stat-label">Total Events</div>
            <div class="coco-stat-value" x-text="summary.total_events"></div>
        </div>
        
        <div class="coco-stat-card">
            <div class="coco-stat-label">Avg Revenue per Staff</div>
            <div class="coco-stat-value" x-text="summary.avg_revenue_per_staff"></div>
        </div>
        
        <div class="coco-stat-card">
            <div class="coco-stat-label">Avg Forecast Accuracy</div>
            <div class="coco-stat-value" x-text="summary.avg_forecast_accuracy"></div>
        </div>
        
        <div class="coco-stat-card">
            <div class="coco-stat-label">Best Performing Event</div>
            <div class="coco-stat-value" x-text="summary.best_event"></div>
        </div>
    </div>
    
    <!-- Reports Table -->
    <div class="coco-card" style="padding: 0;">
        <div class="coco-card-header">
            <h2 class="coco-card-title">Event Reports</h2>
            <button @click="exportReports" class="coco-btn coco-btn-secondary" :disabled="loading">
                <span x-show="!loading">Export CSV</span>
                <span x-show="loading">Exporting...</span>
            </button>
        </div>
        
        <div class="coco-table-container">
            <table class="coco-table">
                <thead>
                    <tr>
                        <th>Event</th>
                        <th>Date</th>
                        <th>Venue</th>
                        <th>Attendance</th>
                        <th>Revenue</th>
                        <th>Staff</th>
                        <th>Rev/Staff</th>
                        <th>Forecast Acc</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <template x-for="report in reports" :key="report.id">
                        <tr>
                            <td>
                                <div class="coco-event-title" x-text="report.event_title"></div>
                            </td>
                            <td x-text="formatDate(report.event_date)"></td>
                            <td x-text="report.venue_name || 'Unknown'"></td>
                            <td x-text="report.metrics?.attendance || 'N/A'"></td>
                            <td x-text="'£' + (report.metrics?.revenue || 0)"></td>
                            <td x-text="report.metrics?.total_staff || 'N/A'"></td>
                            <td x-text="'£' + (report.metrics?.revenue_per_staff || 0)"></td>
                            <td>
                                <span :class="getAccuracyClass(report.metrics?.forecast_accuracy)" 
                                      x-text="(report.metrics?.forecast_accuracy || 0) + '%'"></span>
                            </td>
                            <td>
                                <button @click="viewReport(report)" class="coco-btn coco-btn-sm coco-btn-primary">
                                    View
                                </button>
                            </td>
                        </tr>
                    </template>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Report Detail Modal -->
    <div x-show="showReportModal" class="coco-modal-overlay" @click="showReportModal = false">
        <div class="coco-modal" @click.stop>
            <div class="coco-modal-header">
                <h2 class="coco-modal-title" x-text="selectedReport?.event_title"></h2>
                <p style="font-size: 13px; color: #6B7280; margin: 4px 0 0 0;" x-text="formatDate(selectedReport?.event_date)"></p>
            </div>
            
            <div class="coco-modal-body" x-show="selectedReport">
                <!-- Performance Metrics -->
                <div class="coco-card" style="margin-bottom: 20px;">
                    <h3 class="coco-card-title">Performance Metrics</h3>
                    <div class="coco-stats-grid">
                        <div class="coco-stat-card">
                            <div class="coco-stat-label">Revenue per Staff</div>
                            <div class="coco-stat-value" x-text="'£' + (selectedReport?.metrics?.revenue_per_staff || 0)"></div>
                        </div>
                        
                        <div class="coco-stat-card">
                            <div class="coco-stat-label">Forecast Accuracy</div>
                            <div class="coco-stat-value" x-text="(selectedReport?.metrics?.forecast_accuracy || 0) + '%'"></div>
                        </div>
                        
                        <div class="coco-stat-card">
                            <div class="coco-stat-label">Staff Efficiency</div>
                            <div class="coco-stat-value" x-text="(selectedReport?.metrics?.staff_efficiency || 0) + '%'"></div>
                        </div>
                        
                        <div class="coco-stat-card">
                            <div class="coco-stat-label">vs Historical</div>
                            <div class="coco-stat-value" x-text="(selectedReport?.metrics?.vs_historical || 0) + '%'"></div>
                        </div>
                    </div>
                </div>
                
                <!-- Recommendations -->
                <div class="coco-card" x-show="selectedReport?.recommendations?.length > 0">
                    <h3 class="coco-card-title">Recommendations</h3>
                    <ul>
                        <template x-for="rec in selectedReport?.recommendations" :key="rec">
                            <li x-text="rec"></li>
                        </template>
                    </ul>
                </div>
            </div>
            
            <div class="coco-modal-footer">
                <button @click="showReportModal = false" class="coco-btn coco-btn-secondary">
                    Close
                </button>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('alpine:init', () => {
    Alpine.data('reportsDashboard', () => ({
        reports: [],
        venues: [],
        summary: {
            total_events: 0,
            avg_revenue_per_staff: '£0',
            avg_forecast_accuracy: '0%',
            best_event: 'N/A'
        },
        filters: {
            venue_id: '',
            start_date: '',
            end_date: '',
            metric: 'revenue_per_staff'
        },
        loading: false,
        showReportModal: false,
        selectedReport: null,
        
        async init() {
            await this.loadVenues();
            await this.loadReports();
        },
        
        async loadVenues() {
            try {
                this.venues = await API.getVenues();
            } catch (error) {
                console.error('Failed to load venues:', error);
            }
        },
        
        async loadReports() {
            this.loading = true;
            try {
                const params = new URLSearchParams();
                if (this.filters.venue_id) params.append('venue_id', this.filters.venue_id);
                if (this.filters.start_date) params.append('start_date', this.filters.start_date);
                if (this.filters.end_date) params.append('end_date', this.filters.end_date);
                if (this.filters.metric) params.append('metric', this.filters.metric);
                
                const response = await fetch(`${cocoOpsData.restUrl}reports?${params}`, {
                    headers: {
                        'X-WP-Nonce': cocoOpsData.nonce
                    }
                });
                
                const data = await response.json();
                this.reports = data.reports || [];
                this.summary = data.summary || this.summary;
            } catch (error) {
                console.error('Failed to load reports:', error);
            } finally {
                this.loading = false;
            }
        },
        
        async exportReports() {
            this.loading = true;
            try {
                const params = new URLSearchParams();
                if (this.filters.venue_id) params.append('venue_id', this.filters.venue_id);
                if (this.filters.start_date) params.append('start_date', this.filters.start_date);
                if (this.filters.end_date) params.append('end_date', this.filters.end_date);
                
                const response = await fetch(`${cocoOpsData.restUrl}reports/export?${params}`, {
                    headers: {
                        'X-WP-Nonce': cocoOpsData.nonce
                    }
                });
                
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `event-reports-${new Date().toISOString().split('T')[0]}.csv`;
                a.click();
                window.URL.revokeObjectURL(url);
            } catch (error) {
                console.error('Failed to export reports:', error);
                alert('Failed to export reports');
            } finally {
                this.loading = false;
            }
        },
        
        viewReport(report) {
            this.selectedReport = report;
            this.showReportModal = true;
        },
        
        getVenueName(venueId) {
            const venue = this.venues.find(v => v.id == venueId);
            return venue ? venue.name : 'Unknown';
        },
        
        getAccuracyClass(accuracy) {
            if (accuracy >= 90) return 'text-green-600';
            if (accuracy >= 80) return 'text-yellow-600';
            return 'text-red-600';
        },
        
        formatDate(dateString) {
            if (!dateString) return 'N/A';
            return new Date(dateString).toLocaleDateString();
        }
    }));
});
</script>
