<?php
if (!defined('ABSPATH')) exit;

// Get some diagnostic info
$total_events = wp_count_posts('tribe_events');
$recent_events = get_posts([
    'post_type' => 'tribe_events',
    'post_status' => 'publish',
    'posts_per_page' => 5,
    'orderby' => 'date',
    'order' => 'DESC'
]);

// Get venues using get_posts() method
$venues = [];
$venue_error = null;

try {
    $venues = get_posts([
        'post_type'      => 'tribe_venue',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'orderby'        => 'title',
        'order'          => 'ASC',
    ]);
    
    if (empty($venues)) {
        $venue_error = 'No venues found. Check if venues exist in Events → Venues.';
    }
} catch (Exception $e) {
    $venue_error = $e->getMessage();
    $venues = [];
}

$tribe_active = class_exists('Tribe__Events__Main');
?>

<div class="wrap coco-ops-page">
    <div class="coco-ops-container">
        <!-- Header -->
        <div class="coco-ops-header">
            <h1 class="coco-ops-title">🔍 Debug Information</h1>
            <p style="color: #6B7280; margin-top: 8px;">
                Diagnostic information to help troubleshoot the plugin
            </p>
        </div>

        <!-- Quick Actions -->
        <div class="coco-card" style="background: #EFF6FF; border-left: 4px solid #3B82F6;">
            <h2 class="coco-card-title" style="color: #1E40AF;">⚡ Quick Actions</h2>
            <div style="display: flex; gap: 12px; flex-wrap: wrap;">
                <button 
                    onclick="triggerSnapshots()" 
                    class="button button-primary"
                    style="background: #3B82F6; border-color: #3B82F6;">
                    🔄 Trigger Snapshots Now
                </button>
                <span id="snapshot-status" style="line-height: 32px; color: #1E40AF;"></span>
            </div>
            <p style="font-size: 13px; color: #1E40AF; margin: 12px 0 0 0;">
                This will collect ticket/table sales data for all upcoming events.
            </p>
        </div>
        
        <script>
        function triggerSnapshots() {
            const btn = event.target;
            const status = document.getElementById('snapshot-status');
            
            btn.disabled = true;
            btn.textContent = '⏳ Running...';
            status.textContent = '';
            
            fetch(cocoOpsData.restUrl + 'snapshots/trigger', {
                method: 'POST',
                credentials: 'same-origin',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': cocoOpsData.nonce
                }
            })
            .then(response => response.json())
            .then(data => {
                btn.disabled = false;
                btn.textContent = '🔄 Trigger Snapshots Now';
                
                if (data.success) {
                    status.textContent = '✅ Snapshots completed successfully!';
                    status.style.color = '#059669';
                    
                    setTimeout(() => {
                        status.textContent = '';
                    }, 5000);
                } else {
                    status.textContent = '❌ Error: ' + (data.message || 'Unknown error');
                    status.style.color = '#DC2626';
                }
            })
            .catch(error => {
                btn.disabled = false;
                btn.textContent = '🔄 Trigger Snapshots Now';
                status.textContent = '❌ Error: ' + error.message;
                status.style.color = '#DC2626';
            });
        }
        </script>

        <!-- Plugin Status -->
        <div class="coco-card">
            <h2 class="coco-card-title">Plugin Status</h2>
            <table style="width: 100%; font-size: 14px;">
                <tr>
                    <td style="padding: 8px; font-weight: 600;">Ops Forecast Plugin</td>
                    <td style="padding: 8px;">
                        <span class="coco-badge coco-badge-success">Active</span>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 8px; font-weight: 600;">The Events Calendar</td>
                    <td style="padding: 8px;">
                        <?php if ($tribe_active): ?>
                            <span class="coco-badge coco-badge-success">Active</span>
                        <?php else: ?>
                            <span class="coco-badge coco-badge-gray">Not Active</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 8px; font-weight: 600;">Event Tickets Plus</td>
                    <td style="padding: 8px;">
                        <?php if (class_exists('Tribe__Tickets__Main')): ?>
                            <span class="coco-badge coco-badge-success">Active</span>
                        <?php else: ?>
                            <span class="coco-badge coco-badge-gray">Not Active</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 8px; font-weight: 600;">WooCommerce</td>
                    <td style="padding: 8px;">
                        <?php if (class_exists('WooCommerce')): ?>
                            <span class="coco-badge coco-badge-success">Active</span>
                        <?php else: ?>
                            <span class="coco-badge coco-badge-gray">Not Active</span>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Events Information -->
        <div class="coco-card">
            <h2 class="coco-card-title">Events Data</h2>
            <table style="width: 100%; font-size: 14px;">
                <tr>
                    <td style="padding: 8px; font-weight: 600;">Total Events</td>
                    <td style="padding: 8px;">
                        Published: <?php echo $total_events->publish ?? 0; ?> | 
                        Draft: <?php echo $total_events->draft ?? 0; ?> | 
                        Private: <?php echo $total_events->private ?? 0; ?>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 8px; font-weight: 600;">Recent Events</td>
                    <td style="padding: 8px;">
                        <?php if (!empty($recent_events)): ?>
                            <ul style="margin: 0; padding-left: 20px;">
                                <?php foreach ($recent_events as $event): ?>
                                    <li>
                                        <a href="<?php echo get_edit_post_link($event->ID); ?>" target="_blank">
                                            <?php echo esc_html($event->post_title); ?>
                                        </a>
                                        (<?php echo get_post_meta($event->ID, '_EventStartDate', true) ?: 'No start date'; ?>)
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php else: ?>
                            <span style="color: #6B7280;">No events found</span>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Venues Information -->
        <div class="coco-card">
            <h2 class="coco-card-title">Venues Data</h2>
            <table style="width: 100%; font-size: 14px;">
                <tr>
                    <td style="padding: 8px; font-weight: 600;">Total Venues</td>
                    <td style="padding: 8px;">
                        <?php if ($venue_error): ?>
                            <span class="coco-badge coco-badge-gray">Error: <?php echo esc_html($venue_error); ?></span>
                        <?php else: ?>
                            <?php echo count($venues); ?> venues found
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 8px; font-weight: 600;">Venue List</td>
                    <td style="padding: 8px;">
                        <?php if ($venue_error): ?>
                            <div style="padding: 12px; background: #FEF2F2; border: 1px solid #FECACA; border-radius: 6px; color: #DC2626;">
                                <strong>Error loading venues:</strong> <?php echo esc_html($venue_error); ?>
                                <br><br>
                                <strong>Possible solutions:</strong>
                                <ul style="margin: 8px 0; padding-left: 20px;">
                                    <li>Check if The Events Calendar is properly configured</li>
                                    <li>Try deactivating and reactivating The Events Calendar</li>
                                    <li>Check if venues exist in Events → Venues</li>
                                </ul>
                            </div>
                        <?php elseif (!empty($venues)): ?>
                            <ul style="margin: 0; padding-left: 20px;">
                                <?php foreach ($venues as $venue): ?>
                                    <li>
                                        <strong><?php echo esc_html($venue->post_title ?? $venue->name); ?></strong> 
                                        <?php if (isset($venue->count)): ?>
                                            (<?php echo $venue->count; ?> events)
                                        <?php endif; ?>
                                        <br>
                                        <small style="color: #6B7280;">
                                            ID: <?php echo $venue->ID ?? $venue->id; ?> | 
                                            Slug: <?php echo esc_html($venue->post_name ?? $venue->slug); ?>
                                        </small>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php else: ?>
                            <span style="color: #6B7280;">No venues found</span>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Database Tables -->
        <div class="coco-card">
            <h2 class="coco-card-title">Database Tables</h2>
            <?php
            global $wpdb;
            $tables = [
                'coco_event_features',
                'coco_event_snapshots', 
                'coco_event_planning',
                'coco_staffing_recs',
                'coco_event_actuals',
                'coco_rulesets'
            ];
            ?>
            <table style="width: 100%; font-size: 14px;">
                <?php foreach ($tables as $table): ?>
                    <?php
                    $table_name = $wpdb->prefix . $table;
                    $exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") == $table_name;
                    $count = $exists ? $wpdb->get_var("SELECT COUNT(*) FROM $table_name") : 0;
                    ?>
                    <tr>
                        <td style="padding: 8px; font-weight: 600;"><?php echo $table_name; ?></td>
                        <td style="padding: 8px;">
                            <?php if ($exists): ?>
                                <span class="coco-badge coco-badge-success">Exists</span>
                                (<?php echo $count; ?> rows)
                            <?php else: ?>
                                <span class="coco-badge coco-badge-gray">Missing</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>

        <!-- REST API Test -->
        <div class="coco-card">
            <h2 class="coco-card-title">REST API Test</h2>
            <p style="font-size: 14px; color: #6B7280; margin-bottom: 16px;">
                Test these URLs in your browser to check if the API is working:
            </p>
            <ul style="font-size: 13px; line-height: 1.8;">
                <li><strong>Events:</strong> <a href="<?php echo rest_url('coco-ops/v1/events'); ?>" target="_blank"><?php echo rest_url('coco-ops/v1/events'); ?></a></li>
                <li><strong>Venues:</strong> <a href="<?php echo rest_url('coco-ops/v1/venues'); ?>" target="_blank"><?php echo rest_url('coco-ops/v1/venues'); ?></a></li>
                <li><strong>Settings:</strong> <a href="<?php echo rest_url('coco-ops/v1/settings'); ?>" target="_blank"><?php echo rest_url('coco-ops/v1/settings'); ?></a></li>
            </ul>
        </div>

        <!-- Quick Actions -->
        <div class="coco-card" style="background: #F0FDF4; border-color: #86EFAC;">
            <h2 class="coco-card-title">Quick Actions</h2>
            <div style="display: flex; gap: 12px; flex-wrap: wrap;">
                <button onclick="window.location.reload()" class="coco-btn coco-btn-secondary">
                    🔄 Refresh Page
                </button>
                <button onclick="window.open('<?php echo admin_url('edit.php?post_type=tribe_events'); ?>', '_blank')" class="coco-btn coco-btn-secondary">
                    📅 View Events
                </button>
                <button onclick="window.open('<?php echo admin_url('edit-tags.php?taxonomy=tribe_events_venue'); ?>', '_blank')" class="coco-btn coco-btn-secondary">
                    🏢 View Venues
                </button>
                <button onclick="window.open('<?php echo rest_url('coco-ops/v1/events'); ?>', '_blank')" class="coco-btn coco-btn-primary">
                    🔗 Test API
                </button>
            </div>
        </div>
    </div>
</div>
