<?php
/**
 * Populate Event Features for Forecasting
 * 
 * This script extracts features from all events to populate the coco_event_features table.
 * This is required for the forecasting system to work.
 */

// Include WordPress
require_once('../../../wp-config.php');

global $wpdb;

echo "🚀 Populating Event Features for Forecasting\n";
echo "Time: " . current_time('mysql') . "\n\n";

// Set execution time limit
set_time_limit(0);

$start_time = microtime(true);
$processed = 0;
$errors = 0;

try {
    // Get all events that need features
    $events = $wpdb->get_results("
        SELECT p.ID, p.post_title, pm_start.meta_value as start_date, pm_venue.meta_value as venue_id
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        LEFT JOIN {$wpdb->postmeta} pm_venue ON (p.ID = pm_venue.post_id AND pm_venue.meta_key = '_EventVenueID')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        AND pm_start.meta_value IS NOT NULL
        ORDER BY pm_start.meta_value ASC
    ");
    
    echo "📊 Found " . count($events) . " events to process\n\n";
    
    foreach ($events as $event) {
        echo "📅 Processing: {$event->post_title} (ID: {$event->ID})\n";
        
        try {
            // Parse event start date
            $start_date = new DateTime($event->start_date);
            $weekday = $start_date->format('l'); // Monday, Tuesday, etc.
            $hour = (int) $start_date->format('H');
            
            // Determine start bucket
            if ($hour >= 6 && $hour < 12) {
                $start_bucket = 'morning';
            } elseif ($hour >= 12 && $hour < 17) {
                $start_bucket = 'afternoon';
            } elseif ($hour >= 17 && $hour < 22) {
                $start_bucket = 'evening';
            } else {
                $start_bucket = 'night';
            }
            
            // Check if features already exist
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_features WHERE event_id = %d",
                $event->ID
            ));
            
            if ($existing > 0) {
                echo "  ⚠️  Features already exist, skipping\n";
                $processed++;
                continue;
            }
            
            // Insert event features (matching actual table structure)
            $result = $wpdb->insert(
                $wpdb->prefix . 'coco_event_features',
                [
                    'event_id' => $event->ID,
                    'venue_id' => $event->venue_id ?: 'unknown',
                    'start_ts' => $event->start_date,
                    'weekday' => $weekday,
                    'start_bucket' => $start_bucket,
                    'capacity' => 0, // Default capacity
                    'tags_json' => null,
                    'holiday_flag' => 0
                ],
                ['%d', '%s', '%s', '%s', '%s', '%d', '%s', '%d']
            );
            
            if ($result) {
                echo "  ✅ Features created: {$weekday} {$start_bucket} (venue: {$event->venue_id})\n";
                $processed++;
            } else {
                echo "  ❌ Failed to create features: " . $wpdb->last_error . "\n";
                $errors++;
            }
            
        } catch (Exception $e) {
            echo "  ❌ Error processing event {$event->ID}: " . $e->getMessage() . "\n";
            $errors++;
        }
    }
    
    $time_taken = round(microtime(true) - $start_time, 2);
    
    echo "\n🎉 Event Features Population Complete!\n";
    echo "📊 Total processed: {$processed} events\n";
    echo "❌ Errors: {$errors}\n";
    echo "⏱️  Time taken: {$time_taken}s\n";
    
    // Check if we have enough features for forecasting
    $total_features = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_features");
    echo "📈 Total event features: {$total_features}\n";
    
    if ($total_features > 0) {
        echo "✅ Event features are now available for forecasting!\n";
        echo "🔄 Try the auto forecast again - it should now work.\n";
    } else {
        echo "❌ No event features were created. Check for errors above.\n";
    }
    
} catch (Exception $e) {
    echo "❌ Fatal error: " . $e->getMessage() . "\n";
}

echo "\n🏁 Event features population finished\n";
?>
