<?php

if (!defined('ABSPATH')) {
    exit;
}

class Coco_Ops_Snapshot_Cron {
    
    /**
     * Run snapshots for all upcoming events
     */
    public function run_snapshots() {
        // Get all upcoming events
        $upcoming_events = $this->get_upcoming_events();
        
        if (empty($upcoming_events)) {
            return;
        }
        
        $forecasting_engine = new Coco_Ops_Forecasting_Engine();
        
        foreach ($upcoming_events as $event_id) {
            // Sync event features first
            Coco_Ops_Database::sync_event_features($event_id);
            
            // Create snapshot
            $snapshot_id = $this->create_snapshot($event_id);
            
            // Generate forecast for this snapshot
            if ($snapshot_id) {
                $forecast = $forecasting_engine->generate_forecast($event_id, $snapshot_id);
                
                if (!isset($forecast['error']) && isset($forecast['forecast'])) {
                    $forecasting_engine->update_snapshot_forecast($snapshot_id, $forecast['forecast']);
                }
            }
        }
    }
    
    /**
     * Generate post-event snapshots (day after event ends)
     */
    public function generate_post_event_snapshots() {
        global $wpdb;
        
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        
        // Get events that ended yesterday (for post-event snapshots)
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        
        // Query ALL tribe_events that ended yesterday, not just those in coco_event_features
        $events = $wpdb->get_results($wpdb->prepare("
            SELECT p.ID as event_id, pm.meta_value as start_ts
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_EventStartDate'
            WHERE p.post_type = 'tribe_events'
            AND p.post_status = 'publish'
            AND DATE(pm.meta_value) = %s
            ORDER BY pm.meta_value ASC
        ", $yesterday));
        
        $snapshots_created = 0;
        
        foreach ($events as $event) {
            // Check if we already have a post-event snapshot
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_snapshots 
                 WHERE event_id = %d AND snapshot_ts = %s",
                $event->event_id,
                $yesterday . ' 23:59:59'
            ));
            
            if ($existing == 0) {
                $this->create_snapshot($event->event_id, $yesterday . ' 23:59:59');
                $snapshots_created++;
            }
        }
        
        error_log("CocoOps: Generated $snapshots_created post-event snapshots for $yesterday");
        return $snapshots_created;
    }
    
    /**
     * Get all upcoming events from tribe_events
     */
    private function get_upcoming_events() {
        $args = [
            'post_type' => 'tribe_events',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => [
                [
                    'key' => '_EventStartDate',
                    'value' => current_time('mysql'),
                    'compare' => '>',
                    'type' => 'DATETIME'
                ]
            ],
            'fields' => 'ids'
        ];
        
        $query = new WP_Query($args);
        return $query->posts;
    }
    
    /**
     * Create a snapshot for a specific event
     */
    public function create_snapshot($event_id, $snapshot_date = null) {
        global $wpdb;
        
        // Get event start time
        $start_date = get_post_meta($event_id, '_EventStartDate', true);
        if (!$start_date) {
            return false;
        }
        
        $start_datetime = new DateTime($start_date);
        
        if ($snapshot_date) {
            // For post-event snapshots, use the provided date
            $snapshot_datetime = new DateTime($snapshot_date);
            $interval = $start_datetime->diff($snapshot_datetime);
            $days_to_event = $interval->days + ($interval->h / 24);
            if ($interval->invert == 0) {
                $days_to_event = -$days_to_event; // Event is in the past
            }
        } else {
            // For regular snapshots, use current time
            $now = new DateTime();
            $interval = $start_datetime->diff($now);
            $days_to_event = $interval->days + ($interval->h / 24);
            if ($interval->invert == 0) {
                // Event is in the past
                return false;
            }
        }
        
        // Get tickets and tables sold (tables are tickets priced at £100+)
        $ticket_data = $this->get_tickets_and_tables_sold($event_id);
        $tickets_sold = $ticket_data['tickets'];
        $tables_sold = $ticket_data['tables'];
        $revenue_to_date = $ticket_data['revenue'];
        
        error_log("CocoOpsSnapshot: Event {$event_id} - Creating snapshot with: tickets={$tickets_sold}, tables={$tables_sold}, revenue=£{$revenue_to_date}, days_to_event={$days_to_event}");
        
        // Insert snapshot
        $table = $wpdb->prefix . 'coco_event_snapshots';
        
        $snapshot_timestamp = $snapshot_date ? $snapshot_date : current_time('mysql');
        
        $result = $wpdb->insert(
            $table,
            [
                'event_id' => $event_id,
                'snapshot_ts' => $snapshot_timestamp,
                'days_to_event' => $days_to_event,
                'tickets_sold' => $tickets_sold,
                'tables_sold' => $tables_sold,
                'revenue_to_date' => $revenue_to_date
            ],
            ['%d', '%s', '%f', '%d', '%d', '%f']
        );
        
        return $result ? $wpdb->insert_id : false;
    }
    
    /**
     * Get tickets and tables sold for an event
     * Tables are tickets priced at or above the threshold (default £100)
     * Uses HPOS (High-Performance Order Storage) compatible queries
     * 
     * @return array ['tickets' => int, 'tables' => int, 'revenue' => float]
     */
    private function get_tickets_and_tables_sold($event_id) {
        global $wpdb;
        
        // Get table price threshold from settings
        $settings = get_option('coco_ops_settings', ['table_price_threshold' => 100.00]);
        $table_threshold = isset($settings['table_price_threshold']) ? (float) $settings['table_price_threshold'] : 100.00;
        
        error_log("CocoOpsSnapshot: Getting tickets/tables for event {$event_id}, threshold: £{$table_threshold}");
        
        // First check if event has ANY products linked to it
        $product_count = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*)
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_tribe_wooticket_for_event'
            AND meta_value = %d
        ", $event_id));
        
        error_log("CocoOpsSnapshot: Event {$event_id} has {$product_count} products linked");
        
        // Query all completed orders for this event using HPOS-compatible query
        // This matches the exact pattern from coco-table-reports
        $orders = $wpdb->get_results($wpdb->prepare("
            SELECT 
                p_ticket.post_title as ticket_name,
                pm_price.meta_value as ticket_price,
                CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
            FROM {$wpdb->prefix}wc_orders wc_orders
            INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                wc_orders.id = wc_order_items.order_id
                AND wc_order_items.order_item_type = 'line_item'
            )
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                wc_order_items.order_item_id = oim_ticket.order_item_id
                AND oim_ticket.meta_key = '_product_id'
            )
            INNER JOIN {$wpdb->posts} p_ticket ON (
                oim_ticket.meta_value = p_ticket.ID
                AND p_ticket.post_type = 'product'
            )
            INNER JOIN {$wpdb->postmeta} pm_price ON (
                p_ticket.ID = pm_price.post_id
                AND pm_price.meta_key = '_price'
            )
            INNER JOIN {$wpdb->postmeta} pm_event ON (
                p_ticket.ID = pm_event.post_id
                AND pm_event.meta_key = '_tribe_wooticket_for_event'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                wc_order_items.order_item_id = oim_qty.order_item_id
                AND oim_qty.meta_key = '_qty'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                wc_order_items.order_item_id = oim_line_total.order_item_id
                AND oim_line_total.meta_key = '_line_total'
            )
            WHERE pm_event.meta_value = %d
            AND wc_orders.status IN ('wc-completed', 'wc-processing')
            AND wc_orders.status != 'wc-refunded'
            AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
            AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
        ", $event_id));
        
        error_log("CocoOpsSnapshot: Query returned " . count($orders) . " orders");
        
        if ($wpdb->last_error) {
            error_log("CocoOpsSnapshot: SQL Error: " . $wpdb->last_error);
        }
        
        $tickets = 0;
        $tables = 0;
        $revenue = 0;
        
        // Categorize and sum up
        foreach ($orders as $order) {
            $price = (float) $order->ticket_price;
            $qty = (int) $order->quantity;
            $line_total = (float) $order->line_total;
            
            error_log("CocoOpsSnapshot:   - {$order->ticket_name}: £{$price} x{$qty} = £{$line_total}");
            
            // Categorize based on threshold
            if ($price >= $table_threshold) {
                $tables += $qty;
                error_log("CocoOpsSnapshot:     → Counted as TABLE");
            } else {
                $tickets += $qty;
                error_log("CocoOpsSnapshot:     → Counted as TICKET");
            }
            
            $revenue += $line_total;
        }
        
        error_log("CocoOpsSnapshot: Final totals - Tickets: {$tickets}, Tables: {$tables}, Revenue: £{$revenue}");
        
        return [
            'tickets' => $tickets,
            'tables' => $tables,
            'revenue' => $revenue
        ];
    }
    
    /**
     * Manual trigger for testing
     */
    public static function trigger_manual_snapshot() {
        $cron = new self();
        $cron->run_snapshots();
        
        return [
            'success' => true,
            'message' => 'Snapshots completed'
        ];
    }
}

