<?php

if (!defined('ABSPATH')) {
    exit;
}

class Coco_Ops_Rules_Engine {
    
    private $ruleset;
    private $rules;
    
    public function __construct() {
        $this->load_active_ruleset();
    }
    
    private function load_active_ruleset() {
        $this->ruleset = Coco_Ops_Database::get_active_ruleset();
        if ($this->ruleset) {
            $this->rules = $this->parse_yaml($this->ruleset->yaml_content);
        }
    }
    
    /**
     * Simple YAML parser (basic implementation)
     * For production, consider using symfony/yaml or similar
     */
    private function parse_yaml($yaml_content) {
        $rules = [];
        $lines = explode("\n", $yaml_content);
        $current_role = null;
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            // Skip empty lines and comments
            if (empty($line) || strpos($line, '#') === 0 || strpos($line, 'version:') === 0 || strpos($line, 'rules:') === 0) {
                continue;
            }
            
            // Role definition
            if (preg_match('/^(\w+):$/', $line, $matches)) {
                $current_role = $matches[1];
                $rules[$current_role] = [];
                continue;
            }
            
            // Property definition
            if ($current_role && preg_match('/^(\w+):\s*(.+?)(?:\s*#.*)?$/', $line, $matches)) {
                $key = $matches[1];
                $value = trim($matches[2]);
                
                // Remove quotes if present
                $value = trim($value, '"\'');
                
                // Convert to appropriate type
                if (is_numeric($value)) {
                    $value = strpos($value, '.') !== false ? (float) $value : (int) $value;
                }
                
                $rules[$current_role][$key] = $value;
            }
        }
        
        return $rules;
    }
    
    /**
     * Compute staffing recommendations
     * 
     * @param int $attendance Expected attendance
     * @param array $event_features Event characteristics
     * @param int $tables Expected tables (optional)
     * @return array Staffing counts and rationale
     */
    public function compute_staffing($attendance, $event_features = [], $tables = 0) {
        if (!$this->rules) {
            return [
                'error' => 'No active ruleset found',
                'staffing' => []
            ];
        }
        
        $staffing = [];
        $rationale_parts = [];
        
        $roles = ['security', 'bartenders', 'bottle_girls', 'wait_staff', 'managers'];
        
        foreach ($roles as $role) {
            if (!isset($this->rules[$role])) {
                continue;
            }
            
            $role_rules = $this->rules[$role];
            $result = $this->calculate_role_staffing($role, $attendance, $event_features, $role_rules, $tables);
            
            $staffing[$role] = $result['count'];
            $rationale_parts[] = $result['rationale'];
        }
        
        // Apply special weekday substitution rules
        $staffing = $this->apply_weekday_substitution($staffing, $attendance, $event_features);
        
        return [
            'staffing' => $staffing,
            'rationale_md' => implode("\n\n", $rationale_parts),
            'ruleset_version' => $this->ruleset->version ?? '1.0.0'
        ];
    }
    
    /**
     * Apply special weekday substitution rules
     */
    private function apply_weekday_substitution($staffing, $attendance, $event_features) {
        $weekday = $event_features['weekday'] ?? '';
        $holidayFlag = isset($event_features['holiday_flag']) ? (int) $event_features['holiday_flag'] : 0;
        $isPartyWeekend = $this->is_party_weekend($weekday, $holidayFlag);
        
        // Manager substitution: On party-weekdays with < 40 people, manager replaces 1 bartender
        if (!$isPartyWeekend && $attendance < 40) {
            if ($staffing['bartenders']['count'] > 1) {
                $staffing['bartenders']['count'] -= 1;
                $staffing['bartenders']['rationale'] .= "\n- Weekday substitution: -1 bartender (replaced by manager)";
            }
            
            // Ensure manager is always 1
            $staffing['managers']['count'] = 1;
            $staffing['managers']['rationale'] .= "\n- Weekday substitution: Set to 1 (replaces bartender)";
        }
        
        return $staffing;
    }
    
    /**
     * Determine party-weekend: Friday/Saturday are weekend; Thursday/Sunday are weekdays
     * Bank holiday Sundays (holiday_flag = 1) are treated as weekend
     */
    private function is_party_weekend($weekday, $holidayFlag) {
        if ($weekday === 'Friday' || $weekday === 'Saturday') {
            return true;
        }
        if ($weekday === 'Sunday' && (int) $holidayFlag === 1) {
            return true;
        }
        return false;
    }
    
    /**
     * Calculate staffing for a single role
     */
    private function calculate_role_staffing($role, $attendance, $event_features, $role_rules, $tables = 0) {
        $base_ratio = $role_rules['base_ratio'] ?? 100;
        
        // Handle dynamic minimums for security
        $weekday = $event_features['weekday'] ?? '';
        $holidayFlag = isset($event_features['holiday_flag']) ? (int) $event_features['holiday_flag'] : 0;
        $isPartyWeekend = $this->is_party_weekend($weekday, $holidayFlag);
        if ($role === 'security') {
            if ($isPartyWeekend) {
                $floor = $role_rules['weekend_min'] ?? 2;
            } else {
                $floor = $role_rules['weekday_min'] ?? 1;
            }
        } else {
            $floor = $role_rules['min_count'] ?? 1;
        }
        
        $cap = $role_rules['max_count'] ?? 20;
        
        // Base calculation
        $base_count = $attendance * $base_ratio;
        $rationale_lines = [];
        $rationale_lines[] = "### " . ucwords(str_replace('_', ' ', $role)) . ": ";
        $rationale_lines[] = "- Base calculation: {$attendance} guests × {$base_ratio} = " . number_format($base_count, 2);
        
        // Table calculation (for wait_staff and bottle_girls)
        $table_count = 0;
        if ($tables > 0 && isset($role_rules['table_ratio'])) {
            $table_count = $tables * $role_rules['table_ratio'];
            $rationale_lines[] = "- Table calculation: {$tables} tables × {$role_rules['table_ratio']} = " . number_format($table_count, 2);
        }
        
        // Apply modifiers
        $modified_count = $base_count + $table_count;
        $start_bucket = $event_features['start_bucket'] ?? '';
        
        // Party-weekend modifier (Fri/Sat; BH Sunday)
        if (isset($role_rules['weekend_multiplier']) && $isPartyWeekend) {
            $modifier = $role_rules['weekend_multiplier'];
            $modified_count *= $modifier;
            $rationale_lines[] = "- Weekend multiplier: × {$modifier} = " . number_format($modified_count, 2);
        }
        
        // Friday/Saturday modifier
        if (isset($role_rules['friday_saturday_modifier']) && in_array($weekday, ['Friday', 'Saturday'])) {
            $modifier = $role_rules['friday_saturday_modifier'];
            $modified_count *= $modifier;
            $rationale_lines[] = "- Friday/Saturday modifier: × {$modifier} = " . number_format($modified_count, 2);
        }
        
        // Late night modifier
        if (isset($role_rules['late_night_modifier']) && $start_bucket === 'late_night') {
            $modifier = $role_rules['late_night_modifier'];
            $modified_count *= $modifier;
            $rationale_lines[] = "- Late night modifier: × {$modifier} = " . number_format($modified_count, 2);
        }
        
        // Round up
        $rounded_count = ceil($modified_count);
        if ($rounded_count != $modified_count) {
            $rationale_lines[] = "- Rounded up: " . $rounded_count;
        }
        
        // Apply floor
        $final_count = max($rounded_count, $floor);
        if ($final_count > $rounded_count) {
            $rationale_lines[] = "- Floor applied: max({$rounded_count}, {$floor}) = {$final_count}";
        }
        
        // Apply cap
        $final_count = min($final_count, $cap);
        if ($final_count < max($rounded_count, $floor)) {
            $rationale_lines[] = "- Cap applied: min({$rounded_count}, {$cap}) = {$final_count}";
        }
        
        $rationale_lines[] = "**Final: {$final_count} staff**";
        
        return [
            'count' => $final_count,
            'rationale' => implode("\n", $rationale_lines)
        ];
    }
    
    /**
     * Validate YAML content
     */
    public static function validate_yaml($yaml_content) {
        $errors = [];
        
        // Basic validation
        if (empty($yaml_content)) {
            $errors[] = 'YAML content is empty';
            return ['valid' => false, 'errors' => $errors];
        }
        
        // Check for version
        if (strpos($yaml_content, 'version:') === false) {
            $errors[] = 'Missing version field';
        }
        
        // Check for rules section
        if (strpos($yaml_content, 'rules:') === false) {
            $errors[] = 'Missing rules section';
        }
        
        // Check for required roles
        $required_roles = ['security', 'bartenders', 'bottle_girls', 'wait_staff', 'managers'];
        foreach ($required_roles as $role) {
            if (strpos($yaml_content, $role . ':') === false) {
                $errors[] = "Missing role: {$role}";
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
}

