<?php

if (!defined('ABSPATH')) {
    exit;
}

class Coco_Ops_REST_API {
    
    const NAMESPACE = 'coco-ops/v1';
    
    public function register_routes() {
        // Events endpoints
        register_rest_route(self::NAMESPACE, '/events', [
            'methods' => 'GET',
            'callback' => [$this, 'get_events'],
            'permission_callback' => [$this, 'check_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [$this, 'get_event'],
            'permission_callback' => [$this, 'check_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/snapshots', [
            'methods' => 'GET',
            'callback' => [$this, 'get_event_snapshots'],
            'permission_callback' => [$this, 'check_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/cohorts', [
            'methods' => 'GET',
            'callback' => [$this, 'get_event_cohorts'],
            'permission_callback' => [$this, 'check_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/forecast', [
            'methods' => 'GET',
            'callback' => [$this, 'get_event_forecast'],
            'permission_callback' => [$this, 'check_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/compute-staffing', [
            'methods' => 'POST',
            'callback' => [$this, 'compute_staffing'],
            'permission_callback' => [$this, 'check_admin_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/save-plan', [
            'methods' => 'POST',
            'callback' => [$this, 'save_plan'],
            'permission_callback' => [$this, 'check_admin_permission']
        ]);
        
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/save-actuals', [
            'methods' => 'POST',
            'callback' => [$this, 'save_actuals'],
            'permission_callback' => [$this, 'check_admin_permission']
        ]);
        
        // Reports endpoints
        register_rest_route(self::NAMESPACE, '/reports', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_reports'],
                'permission_callback' => [$this, 'check_permission']
            ]
        ]);
        
        register_rest_route(self::NAMESPACE, '/reports/export', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'export_reports'],
                'permission_callback' => [$this, 'check_permission']
            ]
        ]);
        
        // Venues endpoint
        register_rest_route(self::NAMESPACE, '/venues', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_venues'],
                'permission_callback' => [$this, 'check_permission']
            ]
        ]);
        
        // Rulesets endpoints
        register_rest_route(self::NAMESPACE, '/rulesets', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_rulesets'],
                'permission_callback' => [$this, 'check_permission']
            ],
            [
                'methods' => 'POST',
                'callback' => [$this, 'create_ruleset'],
                'permission_callback' => [$this, 'check_admin_permission']
            ]
        ]);
        
        register_rest_route(self::NAMESPACE, '/rulesets/(?P<id>\d+)/activate', [
            'methods' => 'POST',
            'callback' => [$this, 'activate_ruleset'],
            'permission_callback' => [$this, 'check_admin_permission']
        ]);
        
        // Settings endpoint
        register_rest_route(self::NAMESPACE, '/settings', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_settings'],
                'permission_callback' => [$this, 'check_permission']
            ],
            [
                'methods' => 'POST',
                'callback' => [$this, 'save_settings'],
                'permission_callback' => [$this, 'check_admin_permission']
            ]
        ]);
        
        
        // Manual snapshot trigger (for testing)
        register_rest_route(self::NAMESPACE, '/snapshots/trigger', [
            'methods' => 'POST',
            'callback' => [$this, 'trigger_snapshots'],
            'permission_callback' => [$this, 'check_admin_permission']
        ]);
        
        // Get current ticket/table sales for an event (real-time)
        register_rest_route(self::NAMESPACE, '/events/(?P<id>\d+)/current-sales', [
            'methods' => 'GET',
            'callback' => [$this, 'get_current_sales'],
            'permission_callback' => [$this, 'check_permission']
        ]);
    }
    
    public function check_permission() {
        // For now, allow any administrator access
        // This is safe because the plugin pages already check for admin access
        return current_user_can('manage_options') || current_user_can('edit_posts');
    }
    
    public function check_admin_permission() {
        // Allow administrators
        return current_user_can('manage_options');
    }
    
    /**
     * GET /events - List all events with enriched data
     */
    public function get_events($request) {
        global $wpdb;
        
        $venue = $request->get_param('venue');
        $start_date = $request->get_param('start_date');
        $end_date = $request->get_param('end_date');
        $has_actuals = $request->get_param('has_actuals');
        
        // Build WP_Query args
        $args = [
            'post_type' => 'tribe_events',
            'post_status' => 'publish',
            'posts_per_page' => 100,
            'orderby' => 'meta_value',
            'meta_key' => '_EventStartDate',
            'order' => 'ASC'
        ];
        
        if ($start_date || $end_date) {
            $args['meta_query'] = [];
            
            if ($start_date) {
                $args['meta_query'][] = [
                    'key' => '_EventStartDate',
                    'value' => $start_date,
                    'compare' => '>=',
                    'type' => 'DATETIME'
                ];
            }
            
            if ($end_date) {
                $args['meta_query'][] = [
                    'key' => '_EventStartDate',
                    'value' => $end_date,
                    'compare' => '<=',
                    'type' => 'DATETIME'
                ];
            }
        }
        
        $query = new WP_Query($args);
        $events = [];
        
        // Process all posts
        if (!empty($query->posts)) {
            foreach ($query->posts as $post) {
                $event_data = $this->enrich_event_data($post->ID);
                
                // Apply filters
                if ($venue && $event_data['venue_id'] != $venue) {
                    continue;
                }
                
                if ($has_actuals !== null && ($event_data['has_actuals'] ?? false) != $has_actuals) {
                    continue;
                }
                
                $events[] = $event_data;
            }
        }
        
        // If no events found after processing, return debug info
        if (empty($events)) {
            return rest_ensure_response([
                'events' => [],
                'debug' => [
                    'query_found_posts' => $query->found_posts,
                    'post_type_exists' => post_type_exists('tribe_events'),
                    'tribe_events_active' => class_exists('Tribe__Events__Main'),
                    'total_events' => wp_count_posts('tribe_events'),
                    'filters_applied' => [
                        'venue' => $venue,
                        'start_date' => $start_date,
                        'end_date' => $end_date,
                        'has_actuals' => $has_actuals
                    ]
                ]
            ]);
        }
        
        return rest_ensure_response($events);
    }
    
    /**
     * GET /events/{id} - Get single event with full details
     */
    public function get_event($request) {
        $event_id = $request->get_param('id');
        $event_data = $this->enrich_event_data($event_id);
        
        if (!$event_data) {
            return new WP_Error('event_not_found', 'Event not found', ['status' => 404]);
        }
        
        return rest_ensure_response($event_data);
    }
    
    /**
     * Enrich event data with all related information
     */
    private function enrich_event_data($event_id) {
        global $wpdb;
        
        $post = get_post($event_id);
        if (!$post) {
            return null;
        }
        
        $table_features = $wpdb->prefix . 'coco_event_features';
        $table_planning = $wpdb->prefix . 'coco_event_planning';
        $table_staffing = $wpdb->prefix . 'coco_staffing_recs';
        $table_actuals = $wpdb->prefix . 'coco_event_actuals';
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        
        // Get features
        $features = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_features WHERE event_id = %d",
            $event_id
        ));
        
        // Get planning
        $planning = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_planning WHERE event_id = %d",
            $event_id
        ));
        
        // Get latest staffing recommendation
        $staffing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_staffing WHERE event_id = %d ORDER BY created_at DESC LIMIT 1",
            $event_id
        ));
        
        // Get actuals
        $actuals = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_actuals WHERE event_id = %d",
            $event_id
        ));
        
        // Get latest snapshot
        $snapshot = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_snapshots WHERE event_id = %d ORDER BY snapshot_ts DESC LIMIT 1",
            $event_id
        ));
        
        // Get venue info from event meta
        $venue_id = get_post_meta($event_id, '_EventVenueID', true);
        $venue_name = 'Unknown';
        
        if ($venue_id) {
            $venue_post = get_post($venue_id);
            if ($venue_post) {
                $venue_name = $venue_post->post_title;
            }
        }
        
        $start_date = get_post_meta($event_id, '_EventStartDate', true);
        
        return [
            'id' => $event_id,
            'title' => $post->post_title,
            'start_date' => $start_date,
            'venue_id' => $venue_id,
            'venue_name' => $venue_name,
            'capacity' => $features->capacity ?? 0,
            'weekday' => $features->weekday ?? '',
            'start_bucket' => $features->start_bucket ?? '',
            'tags' => $features ? json_decode($features->tags_json, true) : [],
            'planning' => $planning ? [
                'est_attendance' => $planning->est_attendance ?? 0,
                'est_tables' => $planning->est_tables ?? 0,
                'notes' => $planning->notes ?? '',
                'last_estimated_at' => $planning->last_estimated_at ?? null
            ] : null,
            'staffing' => $this->get_staffing_recommendations($event_id, $staffing, $planning, $features),
            'actuals' => $actuals ? [
                'final_attendance' => $actuals->final_attendance,
                'final_revenue' => $actuals->final_revenue,
                'actual_bartenders' => $actuals->actual_bartenders,
                'actual_bottle_girls' => $actuals->actual_bottle_girls,
                'actual_wait' => $actuals->actual_wait,
                'actual_security' => $actuals->actual_security,
                'actual_managers' => $actuals->actual_managers,
                'notes' => $actuals->notes,
                'locked' => !empty($actuals->locked_at)
            ] : null,
            'current_snapshot' => $snapshot ? [
                'tickets_sold' => $snapshot->tickets_sold ?? 0,
                'tables_sold' => $snapshot->tables_sold ?? 0,
                'revenue_snapshot' => $snapshot->revenue_snapshot ?? 0,
                'snapshot_ts' => $snapshot->snapshot_ts ?? null
            ] : null,
            'has_actuals' => !empty($actuals)
        ];
    }
    
    /**
     * GET /events/{id}/snapshots - Get all snapshots for an event
     */
    public function get_event_snapshots($request) {
        global $wpdb;
        $event_id = $request->get_param('id');
        $table = $wpdb->prefix . 'coco_event_snapshots';
        
        $snapshots = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE event_id = %d ORDER BY snapshot_ts ASC",
            $event_id
        ));
        
        return rest_ensure_response($snapshots);
    }
    
    /**
     * GET /events/{id}/cohorts - Get cohort events
     */
    public function get_event_cohorts($request) {
        $event_id = $request->get_param('id');
        $matcher = new Coco_Ops_Cohort_Matcher();
        $cohorts = $matcher->find_cohorts($event_id);
        
        return rest_ensure_response($cohorts);
    }
    
    /**
     * GET /events/{id}/forecast - Get forecast for event
     */
    public function get_event_forecast($request) {
        $event_id = $request->get_param('id');
        $engine = new Coco_Ops_Forecasting_Engine();
        $forecast = $engine->generate_forecast($event_id);
        
        return rest_ensure_response($forecast);
    }
    
    /**
     * POST /events/{id}/compute-staffing - Compute staffing recommendation
     */
    public function compute_staffing($request) {
        $event_id = $request->get_param('id');
        $attendance = $request->get_param('attendance');
        $tables = $request->get_param('tables') ?: 0;
        $use_forecast = $request->get_param('use_forecast');
        
        if ($use_forecast) {
            // Get forecast attendance
            $engine = new Coco_Ops_Forecasting_Engine();
            $forecast = $engine->generate_forecast($event_id);
            
            if (isset($forecast['forecast']['attendance_p50'])) {
                $attendance = $forecast['forecast']['attendance_p50'];
            } else {
                return new WP_Error('no_forecast', 'No forecast available', ['status' => 400]);
            }
        }
        
        if (!$attendance || $attendance <= 0) {
            return new WP_Error('invalid_attendance', 'Invalid attendance value', ['status' => 400]);
        }
        
        // Get event features
        global $wpdb;
        $table_features = $wpdb->prefix . 'coco_event_features';
        $features = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_features WHERE event_id = %d",
            $event_id
        ));
        
        $event_features = [
            'weekday' => $features->weekday ?? '',
            'start_bucket' => $features->start_bucket ?? '',
            'holiday_flag' => isset($features->holiday_flag) ? (int) $features->holiday_flag : 0
        ];
        
        // Compute staffing
        $rules_engine = new Coco_Ops_Rules_Engine();
        $result = $rules_engine->compute_staffing($attendance, $event_features, $tables);
        
        return rest_ensure_response($result);
    }
    
    /**
     * POST /events/{id}/save-plan - Save planning data
     */
    public function save_plan($request) {
        global $wpdb;
        $event_id = $request->get_param('id');
        $est_attendance = $request->get_param('est_attendance');
        $est_tables = $request->get_param('est_tables');
        $notes = $request->get_param('notes');
        $use_forecast = $request->get_param('use_forecast');
        $staffing = $request->get_param('staffing');
        
        $table_planning = $wpdb->prefix . 'coco_event_planning';
        $table_staffing = $wpdb->prefix . 'coco_staffing_recs';
        
        // Save planning
        $planning_data = [
            'est_attendance' => $est_attendance,
            'est_tables' => $est_tables,
            'notes' => $notes,
            'use_forecast' => $use_forecast ? 1 : 0,
            'last_estimated_by' => get_current_user_id(),
            'last_estimated_at' => current_time('mysql')
        ];
        
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_planning WHERE event_id = %d",
            $event_id
        ));
        
        if ($existing > 0) {
            $wpdb->update($table_planning, $planning_data, ['event_id' => $event_id]);
        } else {
            $planning_data['event_id'] = $event_id;
            $wpdb->insert($table_planning, $planning_data);
        }
        
        // Calculate and save staffing recommendation automatically
        if ($est_attendance > 0) {
            // Get event features
            $table_features = $wpdb->prefix . 'coco_event_features';
            $features = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_features WHERE event_id = %d",
                $event_id
            ));
            
            if ($features) {
                $event_features = [
                    'weekday' => $features->weekday ?? '',
                    'start_bucket' => $features->start_bucket ?? ''
                ];
                
                // Calculate staffing using rules engine
                $rules_engine = new Coco_Ops_Rules_Engine();
                $result = $rules_engine->compute_staffing($est_attendance, $event_features, $est_tables ?? 0);
                
                if (isset($result['staffing'])) {
                    // Save staffing recommendation
                    $wpdb->insert($table_staffing, [
                        'event_id' => $event_id,
                        'ruleset_version' => $result['ruleset_version'] ?? '1.0.0',
                        'source' => $use_forecast ? 'forecast_p50' : 'manual_estimate',
                        'bartenders' => $result['staffing']['bartenders'] ?? 0,
                        'bottle_girls' => $result['staffing']['bottle_girls'] ?? 0,
                        'wait_staff' => $result['staffing']['wait_staff'] ?? 0,
                        'security' => $result['staffing']['security'] ?? 0,
                        'managers' => $result['staffing']['managers'] ?? 0,
                        'est_attendance' => $est_attendance,
                        'est_tables' => $est_tables ?? 0,
                        'rationale_md' => $result['rationale_md'] ?? '',
                        'created_at' => current_time('mysql')
                    ]);
                }
            }
        }
        
        return rest_ensure_response(['success' => true, 'message' => 'Plan saved successfully']);
    }
    
    /**
     * POST /events/{id}/save-actuals - Save actuals data
     */
    public function save_actuals($request) {
        global $wpdb;
        $event_id = $request->get_param('id');
        
        $table_actuals = $wpdb->prefix . 'coco_event_actuals';
        
        $actuals_data = [
            'final_attendance' => $request->get_param('final_attendance'),
            'final_revenue' => $request->get_param('final_revenue'),
            'actual_bartenders' => $request->get_param('actual_bartenders'),
            'actual_bottle_girls' => $request->get_param('actual_bottle_girls'),
            'actual_wait' => $request->get_param('actual_wait'),
            'actual_security' => $request->get_param('actual_security'),
            'actual_managers' => $request->get_param('actual_managers'),
            'notes' => $request->get_param('notes'),
            'locked_at' => $request->get_param('locked') ? current_time('mysql') : null,
            'entered_by' => get_current_user_id(),
            'entered_at' => current_time('mysql')
        ];
        
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_actuals WHERE event_id = %d",
            $event_id
        ));
        
        if ($existing > 0) {
            // Check if locked
            $locked = $wpdb->get_var($wpdb->prepare(
                "SELECT locked_at FROM $table_actuals WHERE event_id = %d",
                $event_id
            ));
            
            if ($locked && !current_user_can('administrator')) {
                return new WP_Error('locked', 'Actuals are locked', ['status' => 403]);
            }
            
            $wpdb->update($table_actuals, $actuals_data, ['event_id' => $event_id]);
        } else {
            $actuals_data['event_id'] = $event_id;
            $wpdb->insert($table_actuals, $actuals_data);
        }
        
        // Generate post-event report
        try {
            $report_generator = new Coco_Ops_Post_Event_Report();
            $report = $report_generator->generate_report($event_id, $actuals_data);
            
            if ($report) {
                error_log('CocoOps: Post-event report generated for event ' . $event_id);
            }
        } catch (Exception $e) {
            error_log('CocoOps: Failed to generate post-event report: ' . $e->getMessage());
        }
        
        return rest_ensure_response(['success' => true, 'message' => 'Actuals saved successfully']);
    }
    
    /**
     * GET /rulesets - Get all rulesets
     */
    public function get_rulesets() {
        global $wpdb;
        $table = $wpdb->prefix . 'coco_rulesets';
        
        $rulesets = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC");
        
        return rest_ensure_response($rulesets);
    }
    
    /**
     * POST /rulesets - Create new ruleset
     */
    public function create_ruleset($request) {
        global $wpdb;
        $version = $request->get_param('version');
        $yaml_content = $request->get_param('yaml_content');
        
        // Validate YAML
        $validation = Coco_Ops_Rules_Engine::validate_yaml($yaml_content);
        if (!$validation['valid']) {
            return new WP_Error('invalid_yaml', implode(', ', $validation['errors']), ['status' => 400]);
        }
        
        $table = $wpdb->prefix . 'coco_rulesets';
        
        $result = $wpdb->insert($table, [
            'version' => $version,
            'yaml_content' => $yaml_content,
            'active' => 0
        ]);
        
        if ($result === false) {
            return new WP_Error('insert_failed', 'Failed to create ruleset', ['status' => 500]);
        }
        
        return rest_ensure_response(['success' => true, 'id' => $wpdb->insert_id]);
    }
    
    /**
     * POST /rulesets/{id}/activate - Activate a ruleset
     */
    public function activate_ruleset($request) {
        global $wpdb;
        $ruleset_id = $request->get_param('id');
        $table = $wpdb->prefix . 'coco_rulesets';
        
        // Deactivate all
        $wpdb->update($table, ['active' => 0], ['active' => 1]);
        
        // Activate selected
        $wpdb->update($table, ['active' => 1], ['id' => $ruleset_id]);
        
        return rest_ensure_response(['success' => true, 'message' => 'Ruleset activated']);
    }
    
    /**
     * GET /settings - Get settings
     */
    public function get_settings() {
        $settings = get_option('coco_ops_settings', $this->get_default_settings());
        return rest_ensure_response($settings);
    }
    
    /**
     * POST /settings - Save settings
     */
    public function save_settings($request) {
        $settings = $request->get_json_params();
        update_option('coco_ops_settings', $settings);
        
        return rest_ensure_response(['success' => true, 'message' => 'Settings saved']);
    }
    
    private function get_default_settings() {
        return [
            'snapshot_times' => ['10:00', '18:00'],
            'cohort_sensitivity' => 'medium',
            'enable_forecasting' => true,
            'min_cohort_size' => 3,
            'table_price_threshold' => 100.00
        ];
    }
    
    /**
     * GET /venues - Get venues from The Events Calendar
     */
    public function get_venues() {
        // Use get_posts() to retrieve venues (recommended method)
        $venues = get_posts([
            'post_type'      => 'tribe_venue',
            'posts_per_page' => -1,
            'post_status'    => 'publish',
            'orderby'        => 'title',
            'order'          => 'ASC',
        ]);
        
        if (!empty($venues)) {
            $venue_list = [];
            
            foreach ($venues as $venue) {
                $venue_id = $venue->ID;
                $venue_name = get_the_title($venue_id);
                
                // Count events for this venue
                $event_count = count(get_posts([
                    'post_type' => 'tribe_events',
                    'meta_key' => '_EventVenueID',
                    'meta_value' => $venue_id,
                    'posts_per_page' => -1,
                    'post_status' => 'publish',
                    'fields' => 'ids'
                ]));
                
                $venue_list[] = [
                    'id' => $venue_id,
                    'name' => $venue_name,
                    'slug' => $venue->post_name,
                    'count' => $event_count
                ];
            }
            
            return rest_ensure_response($venue_list);
        }
        
        // Debug information if no venues found
        global $wpdb;
        $venue_posts_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'tribe_venue' AND post_status = 'publish'");
        $all_venue_posts = $wpdb->get_results("SELECT ID, post_title, post_status FROM {$wpdb->posts} WHERE post_type = 'tribe_venue' LIMIT 5");
        
        return rest_ensure_response([
            'venues' => [],
            'debug' => [
                'method_used' => 'get_posts',
                'venue_posts_count' => $venue_posts_count,
                'all_venue_posts' => $all_venue_posts,
                'error' => 'No published venues found using get_posts()'
            ]
        ]);
    }
    
    /**
     * POST /snapshots/trigger - Manually trigger snapshot collection
     */
    public function trigger_snapshots() {
        $result = Coco_Ops_Snapshot_Cron::trigger_manual_snapshot();
        
        return rest_ensure_response([
            'success' => true,
            'message' => 'Snapshots triggered successfully',
            'result' => $result
        ]);
    }
    
    /**
     * GET /events/{id}/current-sales - Get real-time ticket/table/revenue data
     */
    public function get_current_sales($request) {
        global $wpdb;
        
        $event_id = $request->get_param('id');
        
        // Get table price threshold from settings
        $settings = get_option('coco_ops_settings', ['table_price_threshold' => 100.00]);
        $table_threshold = isset($settings['table_price_threshold']) ? (float) $settings['table_price_threshold'] : 100.00;
        
        // Get event start date and calculate days to event
        $event_start = get_post_meta($event_id, '_EventStartDate', true);
        $days_to_event = null;
        if ($event_start) {
            $start_dt = new DateTime($event_start);
            $now_dt = new DateTime();
            $diff = $now_dt->diff($start_dt);
            $days_to_event = ($start_dt > $now_dt) ? $diff->days + ($diff->h / 24) : -($diff->days + ($diff->h / 24));
        }
        
        // Query WooCommerce directly for current sales (same query as snapshot cron)
        $orders = $wpdb->get_results($wpdb->prepare("
            SELECT 
                p_ticket.post_title as ticket_name,
                pm_price.meta_value as ticket_price,
                CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
            FROM {$wpdb->prefix}wc_orders wc_orders
            INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                wc_orders.id = wc_order_items.order_id
                AND wc_order_items.order_item_type = 'line_item'
            )
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                wc_order_items.order_item_id = oim_ticket.order_item_id
                AND oim_ticket.meta_key = '_product_id'
            )
            INNER JOIN {$wpdb->posts} p_ticket ON (
                oim_ticket.meta_value = p_ticket.ID
                AND p_ticket.post_type = 'product'
            )
            INNER JOIN {$wpdb->postmeta} pm_price ON (
                p_ticket.ID = pm_price.post_id
                AND pm_price.meta_key = '_price'
            )
            INNER JOIN {$wpdb->postmeta} pm_event ON (
                p_ticket.ID = pm_event.post_id
                AND pm_event.meta_key = '_tribe_wooticket_for_event'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                wc_order_items.order_item_id = oim_qty.order_item_id
                AND oim_qty.meta_key = '_qty'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                wc_order_items.order_item_id = oim_line_total.order_item_id
                AND oim_line_total.meta_key = '_line_total'
            )
            WHERE pm_event.meta_value = %d
            AND wc_orders.status IN ('wc-completed', 'wc-processing')
            AND wc_orders.status != 'wc-refunded'
            AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
            AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
        ", $event_id));
        
        $tickets = 0;
        $tables = 0;
        $revenue = 0;
        
        // Categorize and sum up
        foreach ($orders as $order) {
            $price = (float) $order->ticket_price;
            $qty = (int) $order->quantity;
            $line_total = (float) $order->line_total;
            
            // Categorize based on threshold
            if ($price >= $table_threshold) {
                $tables += $qty;
            } else {
                $tickets += $qty;
            }
            
            $revenue += $line_total;
        }
        
        return rest_ensure_response([
            'days_to_event' => $days_to_event,
            'tickets_sold' => $tickets,
            'tables_sold' => $tables,
            'revenue_to_date' => $revenue,
            'table_threshold' => $table_threshold
        ]);
    }
    
    /**
     * Get staffing recommendations (from database or calculate on-the-fly)
     */
    private function get_staffing_recommendations($event_id, $staffing, $planning, $features) {
        // If we have existing staffing recommendations, return them
        if ($staffing) {
            return [
                'bartenders' => $staffing->bartenders ?? 0,
                'bottle_girls' => $staffing->bottle_girls ?? 0,
                'wait_staff' => $staffing->wait_staff ?? 0,
                'security' => $staffing->security ?? 0,
                'managers' => $staffing->managers ?? 0,
                'est_attendance' => $staffing->est_attendance ?? 0,
                'est_tables' => $staffing->est_tables ?? 0,
                'rationale_md' => $staffing->rationale_md ?? '',
                'source' => 'saved'
            ];
        }
        
        // If we have planning data, calculate recommendations on-the-fly
        if ($planning && $planning->est_attendance > 0) {
            $event_features = [
                'weekday' => $features->weekday ?? '',
                'start_bucket' => $features->start_bucket ?? ''
            ];
            
            try {
                $rules_engine = new Coco_Ops_Rules_Engine();
                $result = $rules_engine->compute_staffing(
                    $planning->est_attendance, 
                    $event_features, 
                    $planning->est_tables ?? 0
                );
                
                if (isset($result['staffing'])) {
                    return [
                        'bartenders' => $result['staffing']['bartenders'] ?? 0,
                        'bottle_girls' => $result['staffing']['bottle_girls'] ?? 0,
                        'wait_staff' => $result['staffing']['wait_staff'] ?? 0,
                        'security' => $result['staffing']['security'] ?? 0,
                        'managers' => $result['staffing']['managers'] ?? 0,
                        'est_attendance' => $planning->est_attendance,
                        'est_tables' => $planning->est_tables ?? 0,
                        'rationale_md' => $result['rationale_md'] ?? '',
                        'source' => 'calculated'
                    ];
                }
            } catch (Exception $e) {
                // If calculation fails, return zeros
            }
        }
        
        // No planning data or calculation failed - return zeros
        return [
            'bartenders' => 0,
            'bottle_girls' => 0,
            'wait_staff' => 0,
            'security' => 0,
            'managers' => 0,
            'est_attendance' => 0,
            'est_tables' => 0,
            'rationale_md' => '',
            'source' => 'none'
        ];
    }
    
    /**
     * GET /reports - Get event reports
     */
    public function get_reports($request) {
        global $wpdb;
        
        $venue_id = $request->get_param('venue_id');
        $start_date = $request->get_param('start_date');
        $end_date = $request->get_param('end_date');
        $metric = $request->get_param('metric') ?: 'revenue_per_staff';
        
        $table_reports = $wpdb->prefix . 'coco_event_reports';
        
        $where_conditions = ['1=1'];
        $query_params = [];
        
        if ($venue_id) {
            $where_conditions[] = 'venue_id = %s';
            $query_params[] = $venue_id;
        }
        
        if ($start_date) {
            $where_conditions[] = 'event_date >= %s';
            $query_params[] = $start_date;
        }
        
        if ($end_date) {
            $where_conditions[] = 'event_date <= %s';
            $query_params[] = $end_date;
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        
        // Get reports
        $reports = $wpdb->get_results($wpdb->prepare("
            SELECT * FROM $table_reports 
            WHERE $where_clause 
            ORDER BY event_date DESC 
            LIMIT 100
        ", $query_params));
        
        // Process reports to extract metrics
        $processed_reports = [];
        foreach ($reports as $report) {
            $report_data = json_decode($report->report_data, true);
            $metrics = $report_data['metrics'] ?? [];
            $actuals = $report_data['actuals'] ?? [];
            
            // Get venue name
            $venue_name = 'Unknown';
            if ($report->venue_id) {
                $venue_post = get_post($report->venue_id);
                if ($venue_post) {
                    $venue_name = $venue_post->post_title;
                }
            }
            
            // Map the data to match frontend expectations
            $processed_reports[] = [
                'id' => $report->id,
                'event_id' => $report->event_id,
                'event_title' => $report->event_title,
                'event_date' => $report->event_date,
                'venue_id' => $report->venue_id,
                'venue_name' => $venue_name,
                'metrics' => [
                    'attendance' => $actuals['final_attendance'] ?? 'N/A',
                    'revenue' => $actuals['final_revenue'] ?? 0,
                    'total_staff' => ($actuals['actual_bartenders'] ?? 0) + 
                                   ($actuals['actual_bottle_girls'] ?? 0) + 
                                   ($actuals['actual_wait'] ?? 0) + 
                                   ($actuals['actual_security'] ?? 0) + 
                                   ($actuals['actual_managers'] ?? 0),
                    'revenue_per_staff' => $metrics['revenue_per_staff'] ?? 0,
                    'forecast_accuracy' => $metrics['attendance_accuracy'] ?? 0,
                    'staff_efficiency' => $metrics['staff_efficiency_vs_historical'] ?? 0,
                    'vs_historical' => $metrics['attendance_vs_historical'] ?? 0
                ],
                'recommendations' => $report_data['recommendations'] ?? []
            ];
        }
        
        // Calculate summary
        $summary = $this->calculate_reports_summary($processed_reports);
        
        return rest_ensure_response([
            'reports' => $processed_reports,
            'summary' => $summary
        ]);
    }
    
    /**
     * GET /reports/export - Export reports as CSV
     */
    public function export_reports($request) {
        global $wpdb;
        
        $venue_id = $request->get_param('venue_id');
        $start_date = $request->get_param('start_date');
        $end_date = $request->get_param('end_date');
        
        $table_reports = $wpdb->prefix . 'coco_event_reports';
        
        $where_conditions = ['1=1'];
        $query_params = [];
        
        if ($venue_id) {
            $where_conditions[] = 'venue_id = %s';
            $query_params[] = $venue_id;
        }
        
        if ($start_date) {
            $where_conditions[] = 'event_date >= %s';
            $query_params[] = $start_date;
        }
        
        if ($end_date) {
            $where_conditions[] = 'event_date <= %s';
            $query_params[] = $end_date;
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        
        $reports = $wpdb->get_results($wpdb->prepare("
            SELECT * FROM $table_reports 
            WHERE $where_clause 
            ORDER BY event_date DESC
        ", $query_params));
        
        // Generate CSV
        $csv_data = "Event Title,Event Date,Venue ID,Attendance,Revenue,Total Staff,Revenue per Staff,Forecast Accuracy,Staff Efficiency\n";
        
        foreach ($reports as $report) {
            $report_data = json_decode($report->report_data, true);
            $metrics = $report_data['metrics'] ?? [];
            
            $csv_data .= sprintf(
                "%s,%s,%s,%s,%.2f,%s,%.2f,%.1f%%,%.1f%%\n",
                $report->event_title,
                $report->event_date,
                $report->venue_id,
                $metrics['attendance'] ?? 'N/A',
                $metrics['revenue'] ?? 0,
                $metrics['total_staff'] ?? 'N/A',
                $metrics['revenue_per_staff'] ?? 0,
                $metrics['forecast_accuracy'] ?? 0,
                $metrics['staff_efficiency'] ?? 0
            );
        }
        
        // Set headers for CSV download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="event-reports-' . date('Y-m-d') . '.csv"');
        echo $csv_data;
        exit;
    }
    
    /**
     * Calculate summary statistics for reports
     */
    private function calculate_reports_summary($reports) {
        if (empty($reports)) {
            return [
                'total_events' => 0,
                'avg_revenue_per_staff' => '£0',
                'avg_forecast_accuracy' => '0%',
                'best_event' => 'N/A'
            ];
        }
        
        $total_events = count($reports);
        $revenue_per_staff_values = array_filter(array_column($reports, 'metrics'), function($m) {
            return isset($m['revenue_per_staff']) && $m['revenue_per_staff'] > 0;
        });
        $forecast_accuracy_values = array_filter(array_column($reports, 'metrics'), function($m) {
            return isset($m['forecast_accuracy']) && $m['forecast_accuracy'] > 0;
        });
        
        $avg_revenue_per_staff = !empty($revenue_per_staff_values) ? 
            array_sum(array_column($revenue_per_staff_values, 'revenue_per_staff')) / count($revenue_per_staff_values) : 0;
        
        $avg_forecast_accuracy = !empty($forecast_accuracy_values) ? 
            array_sum(array_column($forecast_accuracy_values, 'forecast_accuracy')) / count($forecast_accuracy_values) : 0;
        
        // Find best performing event (highest revenue per staff)
        $best_event = 'N/A';
        $best_revenue_per_staff = 0;
        foreach ($reports as $report) {
            if (isset($report['metrics']['revenue_per_staff']) && 
                $report['metrics']['revenue_per_staff'] > $best_revenue_per_staff) {
                $best_revenue_per_staff = $report['metrics']['revenue_per_staff'];
                $best_event = $report['event_title'];
            }
        }
        
        return [
            'total_events' => $total_events,
            'avg_revenue_per_staff' => '£' . number_format($avg_revenue_per_staff, 2),
            'avg_forecast_accuracy' => number_format($avg_forecast_accuracy, 1) . '%',
            'best_event' => $best_event
        ];
    }
    
}

