<?php

if (!defined('ABSPATH')) {
    exit;
}

class Coco_Ops_Post_Event_Report {
    
    /**
     * Generate post-event report when actuals are saved
     */
    public function generate_report($event_id, $actuals_data) {
        global $wpdb;
        
        // Get event details
        $event = $wpdb->get_row($wpdb->prepare(
            "SELECT p.ID, p.post_title, pm_start.meta_value as start_date, pm_venue.meta_value as venue_id
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm_start ON p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate'
             LEFT JOIN {$wpdb->postmeta} pm_venue ON p.ID = pm_venue.post_id AND pm_venue.meta_key = '_EventVenueID'
             WHERE p.ID = %d",
            $event_id
        ));
        
        if (!$event) {
            return false;
        }
        
        // Get forecast data for comparison
        $forecast_data = $this->get_forecast_data($event_id);
        
        // Get historical averages for venue
        $historical_data = $this->get_historical_averages($event->venue_id, $event->start_date);
        
        // Calculate metrics
        $metrics = $this->calculate_metrics($actuals_data, $forecast_data, $historical_data);
        
        // Generate report
        $report = [
            'event_id' => $event_id,
            'event_title' => $event->post_title,
            'event_date' => $event->start_date,
            'venue_id' => $event->venue_id,
            'generated_at' => current_time('mysql'),
            'actuals' => $actuals_data,
            'forecast' => $forecast_data,
            'metrics' => $metrics,
            'recommendations' => $this->generate_recommendations($metrics)
        ];
        
        // Save report to database
        $this->save_report($report);
        
        // Send notifications if needed
        $this->send_notifications($report);
        
        return $report;
    }
    
    /**
     * Get forecast data for comparison
     */
    private function get_forecast_data($event_id) {
        global $wpdb;
        
        // First try to get manual planning data
        $planning = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}coco_event_planning WHERE event_id = %d",
            $event_id
        ));
        
        // If no manual planning, use auto-forecast
        if (!$planning || !$planning->est_attendance) {
            try {
                $forecasting_engine = new Coco_Ops_Forecasting_Engine();
                $forecast = $forecasting_engine->generate_forecast($event_id);
                
                if ($forecast && isset($forecast['forecast'])) {
                    return [
                        'planned_attendance' => $forecast['forecast']['attendance_p50'] ?? null,
                        'planned_revenue' => $forecast['forecast']['revenue_p50'] ?? null,
                        'recommended_staff' => $forecast['staffing'] ?? null,
                        'source' => 'auto_forecast',
                        'confidence' => $forecast['forecast']['confidence'] ?? 'unknown',
                        'cohort_count' => $forecast['forecast']['cohort_count'] ?? 0
                    ];
                }
            } catch (Exception $e) {
                error_log('CocoOps: Failed to get auto-forecast for report: ' . $e->getMessage());
            }
        }
        
        // Get staffing recommendations
        $staffing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}coco_staffing_recommendations WHERE event_id = %d",
            $event_id
        ));
        
        return [
            'planned_attendance' => $planning ? $planning->est_attendance : null,
            'planned_revenue' => $planning ? $planning->est_revenue : null,
            'recommended_staff' => $staffing ? json_decode($staffing->recommendations, true) : null,
            'source' => 'manual_planning'
        ];
    }
    
    /**
     * Get historical averages for venue
     */
    private function get_historical_averages($venue_id, $event_date) {
        global $wpdb;
        
        // Get last 12 months of events at this venue
        $start_date = date('Y-m-d', strtotime($event_date . ' -12 months'));
        
        $historical = $wpdb->get_results($wpdb->prepare("
            SELECT 
                ea.final_attendance,
                ea.final_revenue,
                ea.actual_bartenders,
                ea.actual_bottle_girls,
                ea.actual_wait,
                ea.actual_security,
                ea.actual_managers
            FROM {$wpdb->prefix}coco_event_actuals ea
            JOIN {$wpdb->prefix}coco_event_features ef ON ea.event_id = ef.event_id
            WHERE ef.venue_id = %s
            AND ef.start_ts >= %s
            AND ef.start_ts < %s
            AND ea.final_attendance > 0
        ", $venue_id, $start_date, $event_date));
        
        if (empty($historical)) {
            return null;
        }
        
        // Calculate averages
        $total_events = count($historical);
        $avg_attendance = array_sum(array_column($historical, 'final_attendance')) / $total_events;
        $avg_revenue = array_sum(array_column($historical, 'final_revenue')) / $total_events;
        $avg_bartenders = array_sum(array_column($historical, 'actual_bartenders')) / $total_events;
        $avg_bottle_girls = array_sum(array_column($historical, 'actual_bottle_girls')) / $total_events;
        $avg_wait = array_sum(array_column($historical, 'actual_wait')) / $total_events;
        $avg_security = array_sum(array_column($historical, 'actual_security')) / $total_events;
        $avg_managers = array_sum(array_column($historical, 'actual_managers')) / $total_events;
        
        return [
            'total_events' => $total_events,
            'avg_attendance' => round($avg_attendance),
            'avg_revenue' => round($avg_revenue, 2),
            'avg_bartenders' => round($avg_bartenders),
            'avg_bottle_girls' => round($avg_bottle_girls),
            'avg_wait' => round($avg_wait),
            'avg_security' => round($avg_security),
            'avg_managers' => round($avg_managers)
        ];
    }
    
    /**
     * Calculate performance metrics
     */
    private function calculate_metrics($actuals, $forecast, $historical) {
        $metrics = [];
        
        // Revenue per staff calculations
        $total_staff = ($actuals['actual_bartenders'] ?: 0) + 
                      ($actuals['actual_bottle_girls'] ?: 0) + 
                      ($actuals['actual_wait'] ?: 0) + 
                      ($actuals['actual_security'] ?: 0) + 
                      ($actuals['actual_managers'] ?: 0);
        
        $metrics['revenue_per_staff'] = $total_staff > 0 ? round($actuals['final_revenue'] / $total_staff, 2) : 0;
        
        // Staff efficiency (staff per 100 attendees)
        $metrics['staff_efficiency'] = $actuals['final_attendance'] > 0 ? 
            round(($total_staff / $actuals['final_attendance']) * 100, 1) : 0;
        
        // Forecast accuracy
        if ($forecast['planned_attendance'] && $forecast['planned_attendance'] > 0) {
            $attendance_variance = (($actuals['final_attendance'] - $forecast['planned_attendance']) / $forecast['planned_attendance']) * 100;
            $metrics['attendance_accuracy'] = round(100 - abs($attendance_variance), 1);
            $metrics['attendance_variance'] = round($attendance_variance, 1);
        } else {
            $metrics['attendance_accuracy'] = 0;
            $metrics['attendance_variance'] = 0;
        }
        
        if ($forecast['planned_revenue'] && $forecast['planned_revenue'] > 0) {
            $revenue_variance = (($actuals['final_revenue'] - $forecast['planned_revenue']) / $forecast['planned_revenue']) * 100;
            $metrics['revenue_accuracy'] = round(100 - abs($revenue_variance), 1);
            $metrics['revenue_variance'] = round($revenue_variance, 1);
        } else {
            $metrics['revenue_accuracy'] = 0;
            $metrics['revenue_variance'] = 0;
        }
        
        // Historical comparison
        if ($historical && $historical['avg_attendance'] > 0) {
            $metrics['attendance_vs_historical'] = round((($actuals['final_attendance'] - $historical['avg_attendance']) / $historical['avg_attendance']) * 100, 1);
            $metrics['revenue_vs_historical'] = round((($actuals['final_revenue'] - $historical['avg_revenue']) / $historical['avg_revenue']) * 100, 1);
            $metrics['staff_efficiency_vs_historical'] = $this->calculate_staff_efficiency($actuals, $historical);
        } else {
            $metrics['attendance_vs_historical'] = 0;
            $metrics['revenue_vs_historical'] = 0;
            $metrics['staff_efficiency_vs_historical'] = 0;
        }
        
        // Capacity utilization (if venue capacity is available)
        $metrics['capacity_utilization'] = $this->calculate_capacity_utilization($actuals['final_attendance']);
        
        return $metrics;
    }
    
    /**
     * Calculate staff efficiency vs historical
     */
    private function calculate_staff_efficiency($actuals, $historical) {
        $actual_staff_per_100 = ($actuals['final_attendance'] > 0) ? 
            (($actuals['actual_bartenders'] + $actuals['actual_bottle_girls'] + $actuals['actual_wait']) / $actuals['final_attendance']) * 100 : 0;
        
        $historical_staff_per_100 = ($historical['avg_attendance'] > 0) ? 
            (($historical['avg_bartenders'] + $historical['avg_bottle_girls'] + $historical['avg_wait']) / $historical['avg_attendance']) * 100 : 0;
        
        return round($actual_staff_per_100 - $historical_staff_per_100, 1);
    }
    
    /**
     * Calculate capacity utilization
     */
    private function calculate_capacity_utilization($attendance) {
        // This would need venue capacity data - for now return null
        return null;
    }
    
    /**
     * Generate recommendations based on metrics
     */
    private function generate_recommendations($metrics) {
        $recommendations = [];
        
        // Revenue per staff recommendations
        if ($metrics['revenue_per_staff'] < 200) {
            $recommendations[] = "Low revenue per staff (£{$metrics['revenue_per_staff']}). Consider reducing staff or increasing revenue.";
        } elseif ($metrics['revenue_per_staff'] > 500) {
            $recommendations[] = "High revenue per staff (£{$metrics['revenue_per_staff']}). Consider adding more staff for better service.";
        }
        
        // Forecast accuracy recommendations
        if ($metrics['attendance_accuracy'] < 80) {
            $recommendations[] = "Low attendance forecast accuracy ({$metrics['attendance_accuracy']}%). Review forecasting model.";
        }
        
        if ($metrics['revenue_accuracy'] < 80) {
            $recommendations[] = "Low revenue forecast accuracy ({$metrics['revenue_accuracy']}%). Review pricing strategy.";
        }
        
        // Staff efficiency recommendations
        if ($metrics['staff_efficiency_vs_historical'] > 10) {
            $recommendations[] = "Staff efficiency below historical average. Consider training or process improvements.";
        }
        
        return $recommendations;
    }
    
    /**
     * Save report to database
     */
    private function save_report($report) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'coco_event_reports';
        
        // Create table if it doesn't exist
        $this->create_reports_table();
        
        $wpdb->insert($table_name, [
            'event_id' => $report['event_id'],
            'event_title' => $report['event_title'],
            'event_date' => $report['event_date'],
            'venue_id' => $report['venue_id'],
            'report_data' => json_encode($report),
            'generated_at' => $report['generated_at']
        ]);
    }
    
    /**
     * Create reports table
     */
    private function create_reports_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'coco_event_reports';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id int(11) NOT NULL AUTO_INCREMENT,
            event_id int(11) NOT NULL,
            event_title varchar(255) NOT NULL,
            event_date datetime NOT NULL,
            venue_id varchar(50) NOT NULL,
            report_data longtext NOT NULL,
            generated_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY event_id (event_id),
            KEY venue_id (venue_id),
            KEY event_date (event_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Send notifications
     */
    private function send_notifications($report) {
        // For now, just log the report generation
        error_log('CocoOps: Post-event report generated for event ' . $report['event_id']);
        
        // TODO: Add email notifications, Slack integration, etc.
    }
}
