<?php

if (!defined('ABSPATH')) {
    exit;
}

class Coco_Ops_Forecasting_Engine {
    
    private $cohort_matcher;
    
    public function __construct() {
        $this->cohort_matcher = new Coco_Ops_Cohort_Matcher();
    }
    
    /**
     * Generate forecast for a specific event based on current snapshot
     * 
     * @param int $event_id Event ID
     * @param int $snapshot_id Optional specific snapshot ID
     * @return array Forecast data
     */
    public function generate_forecast($event_id, $snapshot_id = null) {
        global $wpdb;
        
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        $table_features = $wpdb->prefix . 'coco_event_features';
        
        // Get latest snapshot if not specified
        if (!$snapshot_id) {
            $snapshot = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_snapshots 
                WHERE event_id = %d 
                ORDER BY snapshot_ts DESC 
                LIMIT 1",
                $event_id
            ));
        } else {
            $snapshot = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_snapshots WHERE id = %d",
                $snapshot_id
            ));
        }
        
        // Get event start date and calculate days to event first
        $event_start = get_post_meta($event_id, '_EventStartDate', true);
        $days_to_event = null;
        if ($event_start) {
            $start_dt = new DateTime($event_start);
            $now_dt = new DateTime();
            $diff = $now_dt->diff($start_dt);
            $days_to_event = ($start_dt > $now_dt) ? $diff->days : -$diff->days;
        }
        
        // For current events (within 14 days), use live current sales data instead of snapshots
        $is_current_event = false;
        if ($event_start && $days_to_event !== null) {
            // Use live data for events within 14 days (past or future)
            $is_current_event = ($days_to_event >= -14 && $days_to_event <= 14);
        }
        
        if ($is_current_event) {
            // Take a fresh snapshot for current events to ensure we have the latest data
            $this->create_fresh_snapshot($event_id);
            
            // Use live current sales data for current events
            $current_sales = $this->get_live_current_sales($event_id);
            if ($current_sales) {
                $snapshot = (object) [
                    'tickets_sold' => $current_sales['tickets'],
                    'tables_sold' => $current_sales['tables'],
                    'revenue_to_date' => $current_sales['revenue'],
                    'days_to_event' => $days_to_event
                ];
            }
        }
        
        if (!$snapshot) {
            // No snapshot yet - return basic info
            return [
                'current' => [
                    'days_to_event' => $days_to_event,
                    'tickets_sold' => 0,
                    'tables_sold' => 0,
                    'revenue_to_date' => 0
                ],
                'message' => 'No snapshot data available yet. Snapshots are captured automatically twice daily.',
                'forecast' => null
            ];
        }
        
        // Get event features
        $event_features = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_features WHERE event_id = %d",
            $event_id
        ));
        
        if (!$event_features) {
            return [
                'error' => 'Event features not found',
                'forecast' => null
            ];
        }
        
        // Find cohort events
        $cohorts = $this->cohort_matcher->find_cohorts($event_id);
        
        if (empty($cohorts)) {
            // No cohorts found - use fallback
            $fallback_attendance = (int) ($event_features->capacity * 0.7);
            $fallback_revenue = $snapshot->revenue_to_date * 1.5; // Simple extrapolation
            
            return [
                'current' => [
                    'tickets_sold' => $snapshot->tickets_sold,
                    'tables_sold' => $snapshot->tables_sold,
                    'revenue_to_date' => $snapshot->revenue_to_date,
                    'days_to_event' => $days_to_event
                ],
                'forecast' => [
                    'attendance_p50' => $fallback_attendance,
                    'attendance_p25' => (int) ($fallback_attendance * 0.8),
                    'attendance_p75' => (int) ($fallback_attendance * 1.2),
                    'revenue_p50' => $fallback_revenue,
                    'confidence' => 'low',
                    'cohort_count' => 0,
                    'method' => 'fallback'
                ]
            ];
        }
        
        // Get final outcomes for cohorts
        $cohort_ids = array_column($cohorts, 'event_id');
        
        // Get cohort snapshots at same days-to-event for pacing analysis
        $cohort_snapshots_at_day = $this->cohort_matcher->get_cohort_snapshots_at_day(
            $cohort_ids,
            $snapshot->days_to_event
        );
        
        // Get final ticket sales from snapshots (D-0 or latest snapshot)
        $cohort_final_sales = $this->get_cohort_final_ticket_sales($cohort_ids);
        $cohort_revenue = array_column($cohorts, 'final_revenue');
        
        // Calculate pacing-based forecast
        $pacing_forecast = $this->calculate_pacing_forecast($snapshot, $cohort_snapshots_at_day, $cohort_final_sales);
        
        // Use pacing forecast if available, otherwise fall back to median
        if ($pacing_forecast) {
            $ticket_sales_p50 = $pacing_forecast['p50'];
            $ticket_sales_p25 = $pacing_forecast['p25'];
            $ticket_sales_p75 = $pacing_forecast['p75'];
        } else {
            // Fallback to median of final sales
            $ticket_sales_p50 = $this->calculate_percentile($cohort_final_sales, 50);
            $ticket_sales_p25 = $this->calculate_percentile($cohort_final_sales, 25);
            $ticket_sales_p75 = $this->calculate_percentile($cohort_final_sales, 75);
        }
        
        // Note: Ticket sales forecast can be lower than current sales if snapshot is outdated
        // The rule enforcement will be applied to attendance forecast instead
        
        // Calculate revenue percentile (excluding nulls)
        $cohort_revenue_filtered = array_filter($cohort_revenue, function($v) { return $v !== null; });
        $revenue_p50 = !empty($cohort_revenue_filtered) ? 
            $this->calculate_percentile($cohort_revenue_filtered, 50) : null;
        
        // Calculate pacing
        $pacing = $this->calculate_pacing($snapshot, $cohort_snapshots_at_day);
        
        // Determine confidence based on cohort count
        $confidence = 'medium';
        if (count($cohorts) >= 10) {
            $confidence = 'high';
        } elseif (count($cohorts) < 5) {
            $confidence = 'low';
        }
        
        // Calculate expected attendance based on ticket sales and cohort walk-in model
        $attendance_forecast = $this->calculate_attendance_forecast(
            $ticket_sales_p50,
            $ticket_sales_p25,
            $ticket_sales_p75,
            $snapshot->tables_sold,
            $event_id,
            $cohorts,
            $cohort_final_sales
        );
        
        return [
            'forecast' => [
                'ticket_sales_p50' => (int) $ticket_sales_p50,
                'ticket_sales_p25' => (int) $ticket_sales_p25,
                'ticket_sales_p75' => (int) $ticket_sales_p75,
                'attendance_p50' => $attendance_forecast['p50'],
                'attendance_p25' => $attendance_forecast['p25'],
                'attendance_p75' => $attendance_forecast['p75'],
                'revenue_p50' => $revenue_p50,
                'confidence' => $confidence,
                'cohort_count' => count($cohorts),
                'method' => $pacing_forecast ? 'pacing_based' : 'cohort_median',
                'pacing' => $pacing,
                'walkins' => $attendance_forecast['walkins'] ?? null
            ],
            'current' => [
                'tickets_sold' => $snapshot->tickets_sold,
                'tables_sold' => $snapshot->tables_sold,
                'revenue_to_date' => $snapshot->revenue_to_date,
                'days_to_event' => $snapshot->days_to_event
            ],
            'cohorts' => array_slice($cohorts, 0, 10) // Return top 10 for reference
        ];
    }
    
    /**
     * Get final ticket sales for cohort events from snapshots
     * Prioritizes D+1 snapshots (day after event) for most accurate final sales
     */
    private function get_cohort_final_ticket_sales($cohort_ids) {
        global $wpdb;
        
        if (empty($cohort_ids)) {
            return [];
        }
        
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        $placeholders = implode(',', array_fill(0, count($cohort_ids), '%d'));
        
        // First try to get D+1 snapshots (day after event) - most accurate final sales
        $d1_query = "
            SELECT event_id, tickets_sold
            FROM $table_snapshots
            WHERE event_id IN ($placeholders)
            AND days_to_event < 0
            AND days_to_event >= -1
            AND snapshot_ts = (
                SELECT MAX(snapshot_ts) 
                FROM $table_snapshots s2 
                WHERE s2.event_id = $table_snapshots.event_id
                AND s2.days_to_event < 0
                AND s2.days_to_event >= -1
            )
        ";
        
        $d1_results = $wpdb->get_results($wpdb->prepare($d1_query, $cohort_ids));
        $d1_sales = array_column($d1_results, 'tickets_sold', 'event_id');
        
        // Get events that don't have D+1 snapshots
        $events_with_d1 = array_keys($d1_sales);
        $events_without_d1 = array_diff($cohort_ids, $events_with_d1);
        
        if (!empty($events_without_d1)) {
            // Fallback to latest snapshot for events without D+1
            $placeholders_fallback = implode(',', array_fill(0, count($events_without_d1), '%d'));
            $fallback_query = "
                SELECT event_id, tickets_sold
                FROM $table_snapshots
                WHERE event_id IN ($placeholders_fallback)
                AND snapshot_ts = (
                    SELECT MAX(snapshot_ts) 
                    FROM $table_snapshots s2 
                    WHERE s2.event_id = $table_snapshots.event_id
                )
            ";
            
            $fallback_results = $wpdb->get_results($wpdb->prepare($fallback_query, $events_without_d1));
            $fallback_sales = array_column($fallback_results, 'tickets_sold', 'event_id');
            
            // Merge D+1 and fallback results
            $d1_sales = array_merge($d1_sales, $fallback_sales);
        }
        
        error_log("CocoOps: Final sales data - D+1 snapshots: " . count($events_with_d1) . ", Fallback: " . count($events_without_d1));
        
        return $d1_sales;
    }
    
    /**
     * Calculate pacing-based forecast
     */
    private function calculate_pacing_forecast($current_snapshot, $cohort_snapshots, $cohort_final_sales) {
        if (empty($cohort_snapshots) || empty($cohort_final_sales)) {
            return null;
        }
        
        $current_tickets = $current_snapshot->tickets_sold;
        $current_days = $current_snapshot->days_to_event;
        
        // Calculate pacing ratios for each cohort
        $pacing_ratios = [];
        foreach ($cohort_snapshots as $cohort_snapshot) {
            if (isset($cohort_final_sales[$cohort_snapshot->event_id])) {
                $cohort_tickets_at_day = $cohort_snapshot->tickets_sold;
                $cohort_final_tickets = $cohort_final_sales[$cohort_snapshot->event_id];
                
                if ($cohort_tickets_at_day > 0) {
                    $ratio = $cohort_final_tickets / $cohort_tickets_at_day;
                    $pacing_ratios[] = $ratio;
                }
            }
        }
        
        if (empty($pacing_ratios)) {
            return null;
        }
        
        // Calculate percentiles of pacing ratios
        $p50_ratio = $this->calculate_percentile($pacing_ratios, 50);
        $p25_ratio = $this->calculate_percentile($pacing_ratios, 25);
        $p75_ratio = $this->calculate_percentile($pacing_ratios, 75);
        
        // Apply ratios to current tickets to get forecast
        return [
            'p50' => (int) ($current_tickets * $p50_ratio),
            'p25' => (int) ($current_tickets * $p25_ratio),
            'p75' => (int) ($current_tickets * $p75_ratio)
        ];
    }
    
    /**
     * Calculate expected attendance based on ticket sales, no-show rates, and table capacity
     */
    private function calculate_attendance_forecast($ticket_sales_p50, $ticket_sales_p25, $ticket_sales_p75, $tables_sold, $event_id, $cohorts, $cohort_final_sales) {
        // Build cohort-based walk-in model from events that have both final tickets and final attendance
        $walkin_model = $this->fit_walkin_model($cohorts, $cohort_final_sales);
        $table_attendance = $this->calculate_table_attendance($tables_sold);
        
        // Expected walk-ins using linear model: walk_ins = a + b * final_tickets
        $expected_walkins_p50 = max(0, $walkin_model['a'] + $walkin_model['b'] * $ticket_sales_p50);
        $expected_walkins_p25 = max(0, $walkin_model['a'] + $walkin_model['b'] * $ticket_sales_p25);
        $expected_walkins_p75 = max(0, $walkin_model['a'] + $walkin_model['b'] * $ticket_sales_p75);
        
        $attendance_p50 = $ticket_sales_p50 + $table_attendance + $expected_walkins_p50;
        $attendance_p25 = $ticket_sales_p25 + $table_attendance + $expected_walkins_p25;
        $attendance_p75 = $ticket_sales_p75 + $table_attendance + $expected_walkins_p75;
        
        // CRITICAL: Attendance forecast should NEVER be below paid tickets + table entries
        // This ensures we account for walk-ins and don't underestimate attendance
        $paid_tickets_and_tables = $this->calculate_paid_tickets_and_tables($event_id);
        
        // Debug logging (before applying rule)
        error_log("CocoOps Attendance Rule - Event $event_id:");
        error_log("  Original attendance p50: $attendance_p50");
        error_log("  Paid tickets + tables: $paid_tickets_and_tables");
        
        // Apply the rule only to P50 to maintain meaningful percentile ranges
        // P25 and P75 should show the actual forecast range
        $attendance_p50 = max((int) $attendance_p50, (int) $paid_tickets_and_tables);
        // Keep P25 and P75 as calculated (they can be lower than paid tickets + tables)
        $attendance_p25 = (int) $attendance_p25;
        $attendance_p75 = (int) $attendance_p75;
        
        error_log("  Final attendance p50: $attendance_p50");
        
        return [
            'p50' => (int) $attendance_p50,
            'p25' => (int) $attendance_p25,
            'p75' => (int) $attendance_p75,
            'walkins' => [
                'model' => $walkin_model,
                'expected' => [
                    'p25' => (int) $expected_walkins_p25,
                    'p50' => (int) $expected_walkins_p50,
                    'p75' => (int) $expected_walkins_p75
                ]
            ]
        ];
    }

    /**
     * Fit a simple linear model for walk-ins based on cohorts: walk_ins = a + b * final_tickets
     * Uses cohorts with both final attendance and final tickets
     */
    private function fit_walkin_model($cohorts, $cohort_final_sales) {
        $xs = [];
        $ys = [];
        foreach ($cohorts as $cohort) {
            $eventId = isset($cohort->event_id) ? (int) $cohort->event_id : (int) ($cohort['event_id'] ?? 0);
            $finalTickets = isset($cohort_final_sales[$eventId]) ? (int) $cohort_final_sales[$eventId] : null;
            if ($finalTickets === null) {
                continue;
            }
            // Get actuals for attendance
            $finalAttendance = $this->get_final_attendance($eventId);
            if ($finalAttendance === null) {
                continue;
            }
            $walkIns = $finalAttendance - $finalTickets;
            if ($walkIns < 0) {
                continue;
            }
            $xs[] = $finalTickets;
            $ys[] = $walkIns;
        }
        
        if (count($xs) < 3) {
            // Fallback: use average walk-ins from available pairs or default 0
            $avg = 0;
            if (!empty($ys)) {
                $avg = array_sum($ys) / count($ys);
            }
            return [
                'a' => (float) $avg,
                'b' => 0.0,
                'n' => count($ys),
                'method' => 'average_walkins'
            ];
        }
        
        // Linear regression: y = a + b x
        $n = count($xs);
        $sumX = array_sum($xs);
        $sumY = array_sum($ys);
        $sumXY = 0.0;
        $sumXX = 0.0;
        for ($i = 0; $i < $n; $i++) {
            $sumXY += $xs[$i] * $ys[$i];
            $sumXX += $xs[$i] * $xs[$i];
        }
        $den = ($n * $sumXX - $sumX * $sumX);
        if (abs($den) < 1e-6) {
            $avg = $sumY / $n;
            return [
                'a' => (float) $avg,
                'b' => 0.0,
                'n' => $n,
                'method' => 'average_walkins'
            ];
        }
        $b = ($n * $sumXY - $sumX * $sumY) / $den;
        $a = ($sumY - $b * $sumX) / $n;
        
        return [
            'a' => (float) $a,
            'b' => (float) $b,
            'n' => $n,
            'method' => 'linear_regression'
        ];
    }

    private function get_final_attendance($event_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'coco_event_actuals';
        $val = $wpdb->get_var($wpdb->prepare("SELECT final_attendance FROM $table WHERE event_id = %d", $event_id));
        if ($val === null) {
            return null;
        }
        return (int) $val;
    }
    
    /**
     * Calculate expected attendance from table sales
     */
    private function calculate_table_attendance($tables_sold) {
        if ($tables_sold <= 0) {
            return 0;
        }
        
        // Default table capacity (if we can't parse the description)
        $default_capacity_per_table = 6;
        
        // Try to get table capacity from product descriptions
        $table_capacity = $this->get_table_capacity_from_products();
        
        if ($table_capacity > 0) {
            return $tables_sold * $table_capacity;
        } else {
            return $tables_sold * $default_capacity_per_table;
        }
    }
    
    /**
     * Get table capacity from product descriptions
     */
    private function get_table_capacity_from_products() {
        global $wpdb;
        
        // Look for table products with descriptions like "Entry for 6" or "for 8"
        $query = "
            SELECT p.post_excerpt, p.post_content
            FROM {$wpdb->posts} p
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND (p.post_excerpt LIKE '%Entry for%' OR p.post_content LIKE '%Entry for%')
            LIMIT 10
        ";
        
        $products = $wpdb->get_results($query);
        
        $capacities = [];
        foreach ($products as $product) {
            $description = $product->post_excerpt ?: $product->post_content;
            
            // Look for patterns like "Entry for 6", "for 8", etc.
            if (preg_match('/Entry for (\d+)/i', $description, $matches)) {
                $capacities[] = (int) $matches[1];
            } elseif (preg_match('/for (\d+)/i', $description, $matches)) {
                $capacities[] = (int) $matches[1];
            }
        }
        
        // Return the most common capacity, or default to 6
        if (!empty($capacities)) {
            $capacity_counts = array_count_values($capacities);
            return array_keys($capacity_counts, max($capacity_counts))[0];
        }
        
        return 0; // Will use default
    }
    
    /**
     * Calculate paid tickets + table entries (excludes free tickets)
     */
    private function calculate_paid_tickets_and_tables($event_id) {
        global $wpdb;
        
        // Get detailed sales breakdown for this event using HPOS (same as get_historical_sales_data)
        $sales_data = $wpdb->get_results($wpdb->prepare("
            SELECT 
                pm_price.meta_value as ticket_price,
                CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
            FROM {$wpdb->prefix}wc_orders wc_orders
            INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                wc_orders.id = wc_order_items.order_id
                AND wc_order_items.order_item_type = 'line_item'
            )
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                wc_order_items.order_item_id = oim_ticket.order_item_id
                AND oim_ticket.meta_key = '_product_id'
            )
            INNER JOIN {$wpdb->posts} p_ticket ON (
                oim_ticket.meta_value = p_ticket.ID
                AND p_ticket.post_type = 'product'
            )
            INNER JOIN {$wpdb->postmeta} pm_price ON (
                p_ticket.ID = pm_price.post_id
                AND pm_price.meta_key = '_price'
            )
            INNER JOIN {$wpdb->postmeta} pm_event ON (
                p_ticket.ID = pm_event.post_id
                AND pm_event.meta_key = '_tribe_wooticket_for_event'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                wc_order_items.order_item_id = oim_qty.order_item_id
                AND oim_qty.meta_key = '_qty'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                wc_order_items.order_item_id = oim_line_total.order_item_id
                AND oim_line_total.meta_key = '_line_total'
            )
            WHERE pm_event.meta_value = %d
            AND wc_orders.status IN ('wc-completed', 'wc-processing')
            AND wc_orders.status != 'wc-refunded'
            AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
            AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
        ", $event_id));
        
        $paid_tickets = 0;
        $table_entries = 0;
        $table_threshold = 100; // Tables are tickets priced at £100+
        
        error_log("CocoOps Sales Analysis - Event $event_id:");
        error_log("  Found " . count($sales_data) . " sales records");
        
        foreach ($sales_data as $sale) {
            $price = (float) $sale->ticket_price;
            $qty = (int) $sale->quantity;
            $line_total = (float) $sale->line_total;
            
            if ($price >= $table_threshold) {
                // This is a table - count the table capacity
                $table_capacity = $this->get_table_capacity_from_products();
                $table_entries += $qty * ($table_capacity ?: 6);
                error_log("  Table: price=$price, qty=$qty, capacity=" . ($table_capacity ?: 6));
            } elseif ($line_total > 0) {
                // This is a paid ticket (has revenue)
                $paid_tickets += $qty;
                error_log("  Paid ticket: price=$price, qty=$qty, total=$line_total");
            } else {
                error_log("  Free ticket: price=$price, qty=$qty, total=$line_total");
            }
        }
        
        $result = $paid_tickets + $table_entries;
        error_log("  Final calculation: paid=$paid_tickets, tables=$table_entries, total=$result");
        
        return $result;
    }
    
    /**
     * Create a fresh snapshot for current events (only if event hasn't finished)
     */
    private function create_fresh_snapshot($event_id) {
        global $wpdb;
        
        // Check if event has finished
        $event_start = get_post_meta($event_id, '_EventStartDate', true);
        if (!$event_start) {
            return; // Can't determine if event finished
        }
        
        $start_dt = new DateTime($event_start);
        $now_dt = new DateTime();
        
        // Only create fresh snapshots for events that haven't finished yet
        if ($start_dt <= $now_dt) {
            error_log("CocoOps: Event $event_id has finished, skipping fresh snapshot");
            return;
        }
        
        // Check if we already have a snapshot from today
        $today = current_time('Y-m-d');
        $existing = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
            WHERE event_id = %d AND DATE(snapshot_ts) = %s
        ", $event_id, $today));
        
        if ($existing == 0) {
            // Create a fresh snapshot using the snapshot cron class
            require_once(COCO_OPS_PLUGIN_DIR . 'includes/class-snapshot-cron.php');
            $snapshot_cron = new Coco_Ops_Snapshot_Cron();
            $snapshot_cron->create_snapshot($event_id);
            
            error_log("CocoOps: Created fresh snapshot for event $event_id");
        }
    }
    
    /**
     * Get live current sales data for current events
     */
    private function get_live_current_sales($event_id) {
        global $wpdb;
        
        // Get table price threshold from settings
        $settings = get_option('coco_ops_settings', ['table_price_threshold' => 100.00]);
        $table_threshold = isset($settings['table_price_threshold']) ? (float) $settings['table_price_threshold'] : 100.00;
        
        // Query WooCommerce directly for current sales (same query as current sales API)
        $orders = $wpdb->get_results($wpdb->prepare("
            SELECT 
                p_ticket.post_title as ticket_name,
                pm_price.meta_value as ticket_price,
                CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
            FROM {$wpdb->prefix}wc_orders wc_orders
            INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                wc_orders.id = wc_order_items.order_id
                AND wc_order_items.order_item_type = 'line_item'
            )
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                wc_order_items.order_item_id = oim_ticket.order_item_id
                AND oim_ticket.meta_key = '_product_id'
            )
            INNER JOIN {$wpdb->posts} p_ticket ON (
                oim_ticket.meta_value = p_ticket.ID
                AND p_ticket.post_type = 'product'
            )
            INNER JOIN {$wpdb->postmeta} pm_price ON (
                p_ticket.ID = pm_price.post_id
                AND pm_price.meta_key = '_price'
            )
            INNER JOIN {$wpdb->postmeta} pm_event ON (
                p_ticket.ID = pm_event.post_id
                AND pm_event.meta_key = '_tribe_wooticket_for_event'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                wc_order_items.order_item_id = oim_qty.order_item_id
                AND oim_qty.meta_key = '_qty'
            )
            LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                wc_order_items.order_item_id = oim_line_total.order_item_id
                AND oim_line_total.meta_key = '_line_total'
            )
            WHERE pm_event.meta_value = %d
            AND wc_orders.status IN ('wc-completed', 'wc-processing')
            AND wc_orders.status != 'wc-refunded'
            AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
            AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
        ", $event_id));
        
        $tickets = 0;
        $tables = 0;
        $revenue = 0;
        
        // Categorize and sum up
        foreach ($orders as $order) {
            $price = (float) $order->ticket_price;
            $quantity = (int) $order->quantity;
            $line_total = (float) $order->line_total;
            
            $tickets += $quantity;
            $revenue += $line_total;
            
            // Check if it's a table based on price threshold
            if ($price >= $table_threshold) {
                $tables += $quantity;
            }
        }
        
        return [
            'tickets' => $tickets,
            'tables' => $tables,
            'revenue' => $revenue
        ];
    }
    
    /**
     * Calculate pacing vs cohort
     */
    private function calculate_pacing($current_snapshot, $cohort_snapshots) {
        if (empty($cohort_snapshots)) {
            return null;
        }
        
        $cohort_tickets = array_column($cohort_snapshots, 'tickets_sold');
        $cohort_revenue = array_column($cohort_snapshots, 'revenue_to_date');
        
        $median_tickets = $this->calculate_percentile($cohort_tickets, 50);
        $median_revenue = $this->calculate_percentile(
            array_filter($cohort_revenue, function($v) { return $v > 0; }), 
            50
        );
        
        $tickets_variance = $median_tickets > 0 ? 
            (($current_snapshot->tickets_sold - $median_tickets) / $median_tickets) * 100 : 0;
        
        $revenue_variance = $median_revenue > 0 ? 
            (($current_snapshot->revenue_to_date - $median_revenue) / $median_revenue) * 100 : 0;
        
        return [
            'tickets_sold_vs_cohort' => round($tickets_variance, 1),
            'revenue_vs_cohort' => round($revenue_variance, 1),
            'cohort_median_tickets' => (int) $median_tickets,
            'cohort_median_revenue' => $median_revenue
        ];
    }
    
    /**
     * Calculate percentile of an array
     */
    private function calculate_percentile($array, $percentile) {
        if (empty($array)) {
            return 0;
        }
        
        $array = array_values($array);
        sort($array);
        
        $index = ($percentile / 100) * (count($array) - 1);
        $lower = floor($index);
        $upper = ceil($index);
        
        if ($lower === $upper) {
            return $array[$lower];
        }
        
        $weight = $index - $lower;
        return $array[$lower] * (1 - $weight) + $array[$upper] * $weight;
    }
    
    /**
     * Update snapshot with forecast data
     */
    public function update_snapshot_forecast($snapshot_id, $forecast_data) {
        global $wpdb;
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        
        $wpdb->update(
            $table_snapshots,
            [
                'forecast_attendance_p50' => $forecast_data['attendance_p50'] ?? null,
                'forecast_revenue_p50' => $forecast_data['revenue_p50'] ?? null
            ],
            ['id' => $snapshot_id],
            ['%d', '%f'],
            ['%d']
        );
    }
    
    /**
     * Batch process forecasts for all upcoming events
     */
    public function batch_process_forecasts() {
        global $wpdb;
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        $table_features = $wpdb->prefix . 'coco_event_features';
        
        // Get all upcoming events with recent snapshots
        $query = "
            SELECT DISTINCT s.event_id, MAX(s.id) as latest_snapshot_id
            FROM $table_snapshots s
            INNER JOIN $table_features ef ON s.event_id = ef.event_id
            WHERE ef.start_ts > NOW()
            GROUP BY s.event_id
        ";
        
        $events = $wpdb->get_results($query);
        
        $processed = 0;
        foreach ($events as $event) {
            $result = $this->generate_forecast($event->event_id, $event->latest_snapshot_id);
            
            if (!isset($result['error']) && isset($result['forecast'])) {
                $this->update_snapshot_forecast($event->latest_snapshot_id, $result['forecast']);
                $processed++;
            }
        }
        
        return $processed;
    }
}

