<?php

if (!defined('ABSPATH')) {
    exit;
}

class Coco_Ops_Database {
    
    public static function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Table 1: Event Features
        $table_event_features = $wpdb->prefix . 'coco_event_features';
        $sql_features = "CREATE TABLE IF NOT EXISTS $table_event_features (
            event_id BIGINT UNSIGNED NOT NULL PRIMARY KEY,
            venue_id VARCHAR(50) NOT NULL,
            start_ts DATETIME NOT NULL,
            weekday VARCHAR(10) NOT NULL,
            start_bucket VARCHAR(20) NOT NULL,
            capacity INT NOT NULL DEFAULT 0,
            tags_json TEXT,
            holiday_flag TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_venue_weekday (venue_id, weekday),
            INDEX idx_start_ts (start_ts)
        ) $charset_collate;";
        
        // Table 2: Event Snapshots (critical for forecasting)
        $table_snapshots = $wpdb->prefix . 'coco_event_snapshots';
        $sql_snapshots = "CREATE TABLE IF NOT EXISTS $table_snapshots (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            event_id BIGINT UNSIGNED NOT NULL,
            snapshot_ts DATETIME NOT NULL,
            days_to_event DECIMAL(5,2) NOT NULL,
            tickets_sold INT NOT NULL DEFAULT 0,
            tables_sold INT NOT NULL DEFAULT 0,
            revenue_to_date DECIMAL(12,2) NOT NULL DEFAULT 0,
            forecast_attendance_p50 INT NULL,
            forecast_revenue_p50 DECIMAL(12,2) NULL,
            INDEX idx_event_days (event_id, days_to_event),
            INDEX idx_snapshot_ts (snapshot_ts)
        ) $charset_collate;";
        
        // Table 3: Event Planning (manual estimates)
        $table_planning = $wpdb->prefix . 'coco_event_planning';
        $sql_planning = "CREATE TABLE IF NOT EXISTS $table_planning (
            event_id BIGINT UNSIGNED NOT NULL PRIMARY KEY,
            est_attendance INT NULL,
            est_tables INT NULL,
            notes TEXT,
            use_forecast TINYINT(1) DEFAULT 0,
            last_estimated_by BIGINT UNSIGNED NULL,
            last_estimated_at DATETIME NULL
        ) $charset_collate;";
        
        // Table 4: Staffing Recommendations
        $table_staffing = $wpdb->prefix . 'coco_staffing_recs';
        $sql_staffing = "CREATE TABLE IF NOT EXISTS $table_staffing (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            event_id BIGINT UNSIGNED NOT NULL,
            ruleset_version VARCHAR(20) NOT NULL,
            source ENUM('manual_estimate', 'forecast_p50') NOT NULL,
            bartenders INT NOT NULL DEFAULT 0,
            bottle_girls INT NOT NULL DEFAULT 0,
            wait_staff INT NOT NULL DEFAULT 0,
            security INT NOT NULL DEFAULT 0,
            managers INT NOT NULL DEFAULT 0,
            rationale_md LONGTEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_event_id (event_id)
        ) $charset_collate;";
        
        // Table 5: Event Actuals
        $table_actuals = $wpdb->prefix . 'coco_event_actuals';
        $sql_actuals = "CREATE TABLE IF NOT EXISTS $table_actuals (
            event_id BIGINT UNSIGNED NOT NULL PRIMARY KEY,
            final_attendance INT NOT NULL,
            final_revenue DECIMAL(12,2) NULL,
            actual_bartenders INT NULL,
            actual_bottle_girls INT NULL,
            actual_wait INT NULL,
            actual_security INT NULL,
            actual_managers INT NULL,
            notes LONGTEXT,
            locked_at DATETIME NULL,
            entered_by BIGINT UNSIGNED NULL,
            entered_at DATETIME DEFAULT CURRENT_TIMESTAMP
        ) $charset_collate;";
        
        // Table 6: Rulesets
        $table_rulesets = $wpdb->prefix . 'coco_rulesets';
        $sql_rulesets = "CREATE TABLE IF NOT EXISTS $table_rulesets (
            id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            version VARCHAR(20) UNIQUE NOT NULL,
            yaml_content LONGTEXT NOT NULL,
            active TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_active (active)
        ) $charset_collate;";
        
        // Execute table creation
        dbDelta($sql_features);
        dbDelta($sql_snapshots);
        dbDelta($sql_planning);
        dbDelta($sql_staffing);
        dbDelta($sql_actuals);
        dbDelta($sql_rulesets);
    }
    
    
    /**
     * Get active ruleset
     */
    public static function get_active_ruleset() {
        global $wpdb;
        $table_rulesets = $wpdb->prefix . 'coco_rulesets';
        
        return $wpdb->get_row(
            "SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1"
        );
    }
    
    /**
     * Seed default ruleset with basic staffing ratios
     */
    public static function seed_default_ruleset() {
        global $wpdb;
        $table_rulesets = $wpdb->prefix . 'coco_rulesets';
        
        // Check if ruleset already exists
        $existing = $wpdb->get_var("SELECT COUNT(*) FROM $table_rulesets WHERE active = 1");
        if ($existing > 0) {
            return; // Already seeded
        }
        
        // Default YAML rules
        $yaml_content = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022  # 1 security per 45 people
    weekend_multiplier: 1.15  # +15% on weekends
    weekday_min: 1  # Minimum 1 on weekdays
    weekend_min: 2  # Minimum 2 on weekends
    max_count: 20
    
  bartenders:
    base_ratio: 0.023  # 1 bartender per 43 people
    weekend_multiplier: 1.20  # +20% on weekends
    min_count: 2
    max_count: 25
    
  wait_staff:
    base_ratio: 0.008  # 1 wait staff per 125 people
    weekend_multiplier: 1.50  # +50% on weekends
    min_count: 1
    max_count: 15
    
  bottle_girls:
    base_ratio: 0.010  # 1 bottle girl per 100 people
    weekend_multiplier: 1.15  # +15% on weekends
    min_count: 0
    max_count: 10
    
  managers:
    base_ratio: 0.002  # 1 manager per 500 people
    weekend_multiplier: 1.00  # No weekend adjustment
    min_count: 1
    max_count: 5";
        
        // Insert default ruleset
        $wpdb->insert(
            $table_rulesets,
            [
                'name' => 'Default Staffing Rules v1.0',
                'version' => '1.0.0',
                'yaml_content' => $yaml_content,
                'active' => 1,
                'created_at' => current_time('mysql')
            ],
            ['%s', '%s', '%s', '%d', '%s']
        );
    }
    
    /**
     * Sync event features from tribe_events
     */
    public static function sync_event_features($event_id) {
        global $wpdb;
        
        $event = get_post($event_id);
        if (!$event || $event->post_type !== 'tribe_events') {
            return false;
        }
        
        // Get event start time
        $start_ts = get_post_meta($event_id, '_EventStartDate', true);
        if (!$start_ts) {
            return false;
        }
        
        $start_datetime = new DateTime($start_ts);
        $weekday = $start_datetime->format('l'); // Monday, Tuesday, etc.
        $hour = (int) $start_datetime->format('G');
        
        // Determine time bucket
        $start_bucket = 'late_night';
        if ($hour >= 11 && $hour < 15) {
            $start_bucket = 'lunch';
        } elseif ($hour >= 15 && $hour < 21) {
            $start_bucket = 'dinner';
        }
        
        // Get venue
        $venue_ids = wp_get_post_terms($event_id, 'tribe_events_venue', ['fields' => 'ids']);
        $venue_id = !empty($venue_ids) ? $venue_ids[0] : 0;
        
        // Get capacity (custom field or default)
        $capacity = (int) get_post_meta($event_id, '_EventCapacity', true);
        if ($capacity <= 0) {
            $capacity = 500; // Default capacity
        }
        
        // Get tags
        $tags = wp_get_post_terms($event_id, 'post_tag', ['fields' => 'names']);
        $tags_json = !empty($tags) ? json_encode($tags) : '[]';
        
        // Check if holiday (simple check - can be enhanced)
        $holiday_flag = 0;
        
        $table = $wpdb->prefix . 'coco_event_features';
        
        // Insert or update
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE event_id = %d",
            $event_id
        ));
        
        if ($existing > 0) {
            $wpdb->update(
                $table,
                [
                    'venue_id' => $venue_id,
                    'start_ts' => $start_ts,
                    'weekday' => $weekday,
                    'start_bucket' => $start_bucket,
                    'capacity' => $capacity,
                    'tags_json' => $tags_json,
                    'holiday_flag' => $holiday_flag,
                    'updated_at' => current_time('mysql')
                ],
                ['event_id' => $event_id],
                ['%s', '%s', '%s', '%s', '%d', '%s', '%d', '%s'],
                ['%d']
            );
        } else {
            $wpdb->insert(
                $table,
                [
                    'event_id' => $event_id,
                    'venue_id' => $venue_id,
                    'start_ts' => $start_ts,
                    'weekday' => $weekday,
                    'start_bucket' => $start_bucket,
                    'capacity' => $capacity,
                    'tags_json' => $tags_json,
                    'holiday_flag' => $holiday_flag
                ],
                ['%d', '%s', '%s', '%s', '%s', '%d', '%s', '%d']
            );
        }
        
        return true;
    }
}

