<?php
/**
 * Historical Snapshots Generator
 * Generate snapshots for past events to bootstrap forecasting
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

    echo '<h1>📊 Historical Snapshots Generator</h1>';
    
    
    echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    .warning { color: #F59E0B; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
    .step { background: #F0FDF4; padding: 15px; border-radius: 8px; margin: 10px 0; border-left: 4px solid #10B981; }
    .step h3 { margin: 0 0 10px 0; color: #047857; }
    button { padding: 10px 20px; margin: 10px; background: #3B82F6; color: white; border: none; border-radius: 5px; cursor: pointer; }
    button:hover { background: #2563EB; }
    button:disabled { background: #9CA3AF; cursor: not-allowed; }
    .progress { background: #EFF6FF; padding: 15px; border-radius: 8px; margin: 10px 0; }
    .log { background: #1F2937; color: #F9FAFB; padding: 15px; border-radius: 8px; font-family: monospace; font-size: 12px; max-height: 300px; overflow-y: auto; }
</style>';

global $wpdb;

// Get settings
$settings = get_option('coco_ops_settings', ['table_price_threshold' => 100.00]);
$table_threshold = isset($settings['table_price_threshold']) ? (float) $settings['table_price_threshold'] : 100.00;

echo '<div class="step">';
echo '<h3>🔍 System Status</h3>';

// Check existing snapshots
$existing_snapshots = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots");
    echo '<p><strong>Existing snapshots:</strong> ' . $existing_snapshots . '</p>';
    

// Check past events - get the OLDEST events first (same as backend)
$past_events = $wpdb->get_results("
    SELECT p.ID, p.post_title, pm_start.meta_value as start_date
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
    INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
    WHERE p.post_type = 'tribe_events' 
    AND p.post_status = 'publish'
    AND pm_start.meta_value < NOW()
    ORDER BY pm_start.meta_value ASC
    LIMIT 100
");

echo '<p><strong>Past events found:</strong> ' . count($past_events) . '</p>';
echo '<p><strong>Table price threshold:</strong> £' . $table_threshold . '</p>';

// Debug: Show date range
if (!empty($past_events)) {
    $oldest = end($past_events);
    $newest = $past_events[0];
    echo '<p><strong>Date range:</strong> ' . $oldest->start_date . ' to ' . $newest->start_date . '</p>';
} else {
    echo '<p class="warning">⚠️ No past events found in the past year</p>';
    
    // Debug: Check all events regardless of date
    $all_events = $wpdb->get_results("
        SELECT p.ID, p.post_title, pm_start.meta_value as start_date
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        ORDER BY pm_start.meta_value DESC
        LIMIT 10
    ");
    
    if ($all_events) {
        echo '<p><strong>All events (last 10):</strong></p><ul>';
        foreach ($all_events as $event) {
            echo '<li>' . $event->post_title . ' - ' . $event->start_date . '</li>';
        }
        echo '</ul>';
    }
}

echo '</div>';

echo '<div class="step">';
echo '<h3>🧪 Test Batch (5 Events)</h3>';
echo '<p>Let\'s test with 5 recent past events first (D-14 to D-0 = 15 snapshots per event):</p>';

if (!empty($past_events)) {
    echo '<ul>';
    foreach (array_slice($past_events, 0, 5) as $event) {
        $start_date = new DateTime($event->start_date);
        echo '<li><strong>' . $event->post_title . '</strong> - ' . $start_date->format('Y-m-d H:i') . '</li>';
    }
    echo '</ul>';
    
    echo '<button onclick="startTestBatch()" id="testBtn">🚀 Start Test Batch</button>';
    echo '<div id="progress" class="progress" style="display: none;"></div>';
    echo '<div id="log" class="log" style="display: none;"></div>';
} else {
    echo '<p class="warning">⚠️ No past events found</p>';
}

echo '</div>';

echo '<div class="step">';
echo '<h3>📈 Full Batch Processing</h3>';
echo '<p>After testing, we can process all past events:</p>';
echo '<button onclick="startFullBatch()" id="fullBtn" disabled>📊 Process All Past Events</button>';
echo '<p style="font-size: 13px; color: #6B7280;">This will be enabled after successful test batch</p>';
echo '</div>';

?>

    <script>
    let isProcessing = false;
    
    // Debug: Log initial state
    console.log('Historical Snapshots Generator loaded');
    console.log('AJAX URL:', '<?php echo admin_url('admin-ajax.php'); ?>');
    console.log('Nonce:', '<?php echo wp_create_nonce('coco_ops_historical_snapshots'); ?>');
    

async function startTestBatch() {
    if (isProcessing) return;
    
    isProcessing = true;
    const btn = document.getElementById('testBtn');
    const progress = document.getElementById('progress');
    const log = document.getElementById('log');
    
    btn.disabled = true;
    btn.textContent = '⏳ Processing...';
    progress.style.display = 'block';
    log.style.display = 'block';
    
        try {
            console.log('Starting test batch...');
            
            // Create AbortController for timeout
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout
            
            const response = await fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'coco_ops_generate_historical_snapshots',
                    batch_size: 2,
                    nonce: '<?php echo wp_create_nonce('coco_ops_historical_snapshots'); ?>'
                }),
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            
            console.log('Response received:', response.status, response.statusText);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            console.log('Response data:', data);
        
        if (data.success) {
            progress.innerHTML = `
                <h4 style="color: #059669; margin: 0 0 10px 0;">✅ Test Batch Complete!</h4>
                <p><strong>Events processed:</strong> ${data.data.processed}</p>
                <p><strong>Snapshots created:</strong> ${data.data.snapshots_created}</p>
                <p><strong>Time taken:</strong> ${data.data.time_taken}s</p>
                ${data.data.message ? `<p><strong>Message:</strong> ${data.data.message}</p>` : ''}
            `;
            
            // Enable full batch button
            document.getElementById('fullBtn').disabled = false;
            
            // Show log
            log.innerHTML = data.data.log || 'No detailed log available';
        } else {
            progress.innerHTML = `
                <h4 style="color: #DC2626; margin: 0 0 10px 0;">❌ Error</h4>
                <p>${data.message || 'Unknown error occurred'}</p>
                <pre style="background: #FEE2E2; padding: 10px; border-radius: 4px; margin-top: 10px; font-size: 12px;">${JSON.stringify(data, null, 2)}</pre>
            `;
        }
    } catch (error) {
        console.error('Test batch error:', error);
        if (error.name === 'AbortError') {
            progress.innerHTML = `
                <h4 style="color: #DC2626; margin: 0 0 10px 0;">❌ Request Timeout</h4>
                <p>The request took too long to complete (30 seconds). This might indicate a server issue.</p>
            `;
        } else {
            progress.innerHTML = `
                <h4 style="color: #DC2626; margin: 0 0 10px 0;">❌ Network Error</h4>
                <p>${error.message}</p>
            `;
        }
    } finally {
        isProcessing = false;
        btn.disabled = false;
        btn.textContent = '🚀 Start Test Batch';
    }
}

async function startFullBatch() {
    if (isProcessing) return;
    
    if (!confirm('This will process ALL past events. This may take 30-60 minutes. Continue?')) {
        return;
    }
    
    isProcessing = true;
    const btn = document.getElementById('fullBtn');
    const progress = document.getElementById('progress');
    const log = document.getElementById('log');
    
    btn.disabled = true;
    btn.textContent = '⏳ Processing All Events...';
    progress.style.display = 'block';
    log.style.display = 'block';
    
    try {
        const response = await fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'coco_ops_generate_historical_snapshots',
                batch_size: 20,
                nonce: '<?php echo wp_create_nonce('coco_ops_historical_snapshots'); ?>'
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            progress.innerHTML = `
                <h4 style="color: #059669; margin: 0 0 10px 0;">✅ Full Batch Complete!</h4>
                <p><strong>Events processed:</strong> ${data.processed}</p>
                <p><strong>Snapshots created:</strong> ${data.snapshots_created}</p>
                <p><strong>Time taken:</strong> ${data.time_taken}s</p>
            `;
        } else {
            progress.innerHTML = `
                <h4 style="color: #DC2626; margin: 0 0 10px 0;">❌ Error</h4>
                <p>${data.message}</p>
            `;
        }
    } catch (error) {
        console.error('Full batch error:', error);
        progress.innerHTML = `
            <h4 style="color: #DC2626; margin: 0 0 10px 0;">❌ Network Error</h4>
            <p>${error.message}</p>
            <pre style="background: #FEE2E2; padding: 10px; border-radius: 4px; margin-top: 10px; font-size: 12px;">${error.stack}</pre>
        `;
    } finally {
        isProcessing = false;
        btn.disabled = false;
        btn.textContent = '📊 Process All Past Events';
    }
}
</script>

