<?php
/**
 * Generate Snapshots for Current/Future Events
 * 
 * This script creates snapshots for current and future events
 * so the auto forecast can work.
 */

// Include WordPress
require_once('../../../wp-config.php');

global $wpdb;

echo "🚀 Generating Snapshots for Current/Future Events\n";
echo "Time: " . current_time('mysql') . "\n\n";

// Set execution time limit
set_time_limit(0);

$start_time = microtime(true);
$processed = 0;
$snapshots_created = 0;

try {
    // Get current and future events that need snapshots
    $events = $wpdb->get_results("
        SELECT DISTINCT p.ID, p.post_title, pm_start.meta_value as start_date
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        AND pm_start.meta_value >= NOW()
        ORDER BY pm_start.meta_value ASC
        LIMIT 50
    ");
    
    echo "📊 Found " . count($events) . " current/future events to process\n\n";
    
    foreach ($events as $event) {
        echo "📅 Processing: {$event->post_title} (ID: {$event->ID})\n";
        
        // Check if event has products linked
        $products_linked = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*)
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_tribe_wooticket_for_event'
            AND meta_value = %d
        ", $event->ID));
        
        if ($products_linked == 0) {
            echo "  ⚠️  Skipping - no products linked\n";
            $processed++;
            continue;
        }
        
        // Calculate snapshot dates (D-14 to D-0 for future events)
        $event_start = new DateTime($event->start_date);
        $now = new DateTime();
        $days_to_event = $now->diff($event_start)->days;
        
        // Only create snapshots if event is within 14 days
        if ($days_to_event > 14) {
            echo "  ⚠️  Skipping - event is more than 14 days away\n";
            $processed++;
            continue;
        }
        
        $snapshot_dates = [];
        
        // Generate snapshots from D-14 to D-0
        for ($i = 14; $i >= 0; $i--) {
            $snapshot_date = clone $event_start;
            $snapshot_date->modify("-{$i} days");
            
            // Only create snapshots for dates that have already passed
            if ($snapshot_date <= $now) {
                $snapshot_dates[] = $snapshot_date;
            }
        }
        
        echo "  📸 Creating " . count($snapshot_dates) . " snapshots\n";
        
        $event_snapshots = 0;
        
        foreach ($snapshot_dates as $snapshot_date) {
            // Check if snapshot already exists
            $existing = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
                WHERE event_id = %d AND snapshot_ts = %s
            ", $event->ID, $snapshot_date->format('Y-m-d H:i:s')));
            
            if ($existing > 0) {
                continue; // Skip existing snapshots
            }
            
            // Get sales data up to this snapshot date
            $sales_data = get_historical_sales_data($event->ID, $snapshot_date, 100.00);
            
            // Calculate days to event
            $days_to_event = $event_start->diff($snapshot_date)->days;
            if ($event_start < $snapshot_date) {
                $days_to_event = -$days_to_event;
            }
            
            // Insert snapshot
            $result = $wpdb->insert(
                $wpdb->prefix . 'coco_event_snapshots',
                [
                    'event_id' => $event->ID,
                    'snapshot_ts' => $snapshot_date->format('Y-m-d H:i:s'),
                    'days_to_event' => $days_to_event,
                    'tickets_sold' => $sales_data['tickets'],
                    'tables_sold' => $sales_data['tables'],
                    'revenue_to_date' => $sales_data['revenue']
                ],
                ['%d', '%s', '%d', '%d', '%d', '%f']
            );
            
            if ($result) {
                $event_snapshots++;
                $snapshots_created++;
            }
        }
        
        echo "  ✅ Created {$event_snapshots} snapshots\n";
        $processed++;
    }
    
    $time_taken = round(microtime(true) - $start_time, 2);
    
    echo "\n🎉 Current/Future Snapshots Generation Complete!\n";
    echo "📊 Total processed: {$processed} events\n";
    echo "📸 Total snapshots: {$snapshots_created}\n";
    echo "⏱️  Time taken: {$time_taken}s\n";
    
    if ($snapshots_created > 0) {
        echo "✅ Auto forecast should now work for current/future events!\n";
    } else {
        echo "⚠️  No snapshots were created. Events might be too far in the future.\n";
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}

echo "\n🏁 Current snapshots generation finished\n";
?>
