<?php
// Generate post-event snapshots for all past events
require_once('../../../wp-config.php');

echo "📸 Generating Post-Event Snapshots for All Past Events\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

try {
    require_once('includes/class-snapshot-cron.php');
    $cron = new Coco_Ops_Snapshot_Cron();
    
    // Get all past events (ended more than 1 day ago)
    $past_events = $wpdb->get_results("
        SELECT p.ID as event_id, pm.meta_value as start_ts
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_EventStartDate'
        WHERE p.post_type = 'tribe_events'
        AND p.post_status = 'publish'
        AND pm.meta_value < DATE_SUB(NOW(), INTERVAL 1 DAY)
        ORDER BY pm.meta_value DESC
        LIMIT 100
    ");
    
    echo "Found " . count($past_events) . " past events to process\n\n";
    
    $snapshots_created = 0;
    $processed = 0;
    
    foreach ($past_events as $event) {
        $processed++;
        echo "Processing event $processed/" . count($past_events) . ": Event {$event->event_id}\n";
        
        // Calculate the day after the event
        $event_date = date('Y-m-d', strtotime($event->start_ts));
        $post_event_date = date('Y-m-d', strtotime($event_date . ' +1 day'));
        $snapshot_timestamp = $post_event_date . ' 23:59:59';
        
        // Check if we already have a post-event snapshot
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
             WHERE event_id = %d AND snapshot_ts = %s",
            $event->event_id,
            $snapshot_timestamp
        ));
        
        if ($existing == 0) {
            $result = $cron->create_snapshot($event->event_id, $snapshot_timestamp);
            if ($result) {
                $snapshots_created++;
                echo "  ✅ Created snapshot for D+1\n";
            } else {
                echo "  ❌ Failed to create snapshot\n";
            }
        } else {
            echo "  ⏭️  Snapshot already exists\n";
        }
        
        // Small delay to prevent overwhelming the database
        usleep(100000); // 0.1 second
    }
    
    echo "\n✅ Complete! Created $snapshots_created new post-event snapshots\n";
    
    // Show summary
    $total_snapshots = $wpdb->get_var("
        SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
        WHERE days_to_event < 0
    ");
    
    echo "Total post-event snapshots in database: $total_snapshots\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Script complete.\n";
?>
