<?php
/**
 * Fix missing ruleset
 * Visit: /wp-content/plugins/coco-ops-forecast/fix-rules.php
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔧 Fix Missing Ruleset</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
</style>';

global $wpdb;

// Check if ruleset exists
$table_rulesets = $wpdb->prefix . 'coco_rulesets';
$existing = $wpdb->get_var("SELECT COUNT(*) FROM $table_rulesets WHERE active = 1");

if ($existing > 0) {
    echo '<p class="info">✅ Ruleset already exists!</p>';
    
    // Show current ruleset
    $ruleset = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    echo '<h2>Current Ruleset:</h2>';
    echo '<p><strong>Name:</strong> ' . esc_html($ruleset->name) . '</p>';
    echo '<p><strong>Version:</strong> ' . esc_html($ruleset->version) . '</p>';
    echo '<p><strong>Created:</strong> ' . $ruleset->created_at . '</p>';
    
} else {
    echo '<p class="error">❌ No active ruleset found!</p>';
    echo '<p>Creating default ruleset...</p>';
    
    // Create default ruleset with Cococure's specific requirements
    $yaml_content = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022  # 1 security per 45 people
    weekend_multiplier: 1.15  # +15% on weekends
    weekday_min: 1  # Minimum 1 on weekdays
    weekend_min: 2  # Minimum 2 on weekends
    max_count: 20
    
  bartenders:
    base_ratio: 0.023  # 1 bartender per 43 people
    weekend_multiplier: 1.20  # +20% on weekends
    min_count: 2
    max_count: 4  # Limited by bar space
    
  wait_staff:
    base_ratio: 0.008  # 1 wait staff per 125 people
    weekend_multiplier: 1.50  # +50% on weekends
    min_count: 1
    max_count: 15
    
  bottle_girls:
    base_ratio: 0.010  # 1 bottle girl per 100 people
    weekend_multiplier: 1.15  # +15% on weekends
    min_count: 0
    max_count: 10
    
  managers:
    base_ratio: 0.002  # 1 manager per 500 people
    weekend_multiplier: 1.00  # Always 1 needed
    min_count: 1
    max_count: 1
    weekday_substitution: true  # Managers replace 1 bartender on weekdays when < 40 people";
    
    $result = $wpdb->insert(
        $table_rulesets,
        [
            'name' => 'Default Staffing Rules v1.0',
            'version' => '1.0.0',
            'yaml_content' => $yaml_content,
            'active' => 1,
            'created_at' => current_time('mysql')
        ],
        ['%s', '%s', '%s', '%d', '%s']
    );
    
    if ($result) {
        echo '<p class="success">✅ Default ruleset created successfully!</p>';
        echo '<h2>Cococure Staffing Ratios:</h2>';
        echo '<ul>';
        echo '<li><strong>Security:</strong> 1 per 45 people (2.2%) - Min 1 weekdays, Min 2 weekends</li>';
        echo '<li><strong>Bartenders:</strong> 1 per 43 people (2.3%) - MAX 4 (bar space limit)</li>';
        echo '<li><strong>Wait Staff:</strong> 1 per 125 people (0.8%)</li>';
        echo '<li><strong>Bottle Girls:</strong> 1 per 100 people (1.0%)</li>';
        echo '<li><strong>Managers:</strong> 1 per 500 people (0.2%) - Always 1 needed</li>';
        echo '</ul>';
        echo '<p><strong>Weekend Multipliers:</strong> +15% security, +20% bartenders, +50% wait staff, +15% bottle girls</p>';
        echo '<p><strong>Special Rules:</strong> Managers replace 1 bartender on weekdays when < 40 people expected</p>';
    } else {
        echo '<p class="error">❌ Failed to create ruleset: ' . $wpdb->last_error . '</p>';
    }
}

echo '<hr>';
echo '<p style="color: #666; font-size: 12px;">Delete this file after fixing.</p>';
?>
