<?php
/**
 * Fix performance and add table support
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔧 Fix Performance & Add Table Support</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
</style>';

global $wpdb;

try {
    $table_rulesets = $wpdb->prefix . 'coco_rulesets';
    
    // Get current ruleset
    $current = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    
    if (!$current) {
        echo '<p class="error">❌ No active ruleset found!</p>';
        exit;
    }
    
    echo '<p class="info">📋 Current ruleset ID: ' . $current->id . '</p>';
    
    // New optimized YAML with table support
    $new_yaml = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022
    weekend_multiplier: 1.15
    weekday_min: 1
    weekend_min: 2
    max_count: 20
    
  bartenders:
    base_ratio: 0.023
    weekend_multiplier: 1.20
    min_count: 2
    max_count: 4
    
  wait_staff:
    base_ratio: 0.008
    weekend_multiplier: 1.50
    min_count: 1
    max_count: 15
    table_ratio: 0.5  # 0.5 wait staff per table
    
  bottle_girls:
    base_ratio: 0.010
    weekend_multiplier: 1.15
    min_count: 0
    max_count: 10
    table_ratio: 0.3  # 0.3 bottle girls per table
    
  managers:
    base_ratio: 0.002
    weekend_multiplier: 1.00
    min_count: 1
    max_count: 1
    weekday_substitution: true";
    
    // Update ruleset
    $result = $wpdb->update(
        $table_rulesets,
        ['yaml_content' => $new_yaml],
        ['id' => $current->id],
        ['%s'],
        ['%d']
    );
    
    if ($result !== false) {
        echo '<p class="success">✅ Ruleset updated with table support!</p>';
        
        echo '<h2>🎯 New Features Added:</h2>';
        echo '<ul>';
        echo '<li><strong>Wait Staff:</strong> Now considers tables (0.5 staff per table)</li>';
        echo '<li><strong>Bottle Girls:</strong> Now considers tables (0.3 staff per table)</li>';
        echo '<li><strong>Optimized YAML:</strong> Cleaner format for faster parsing</li>';
        echo '</ul>';
        
        echo '<h3>📊 Example: 200 people + 20 tables on Saturday</h3>';
        echo '<ul>';
        echo '<li><strong>Wait Staff:</strong> (200 × 0.008 × 1.50) + (20 × 0.5) = 2.4 + 10 = 12 staff</li>';
        echo '<li><strong>Bottle Girls:</strong> (200 × 0.010 × 1.15) + (20 × 0.3) = 2.3 + 6 = 8 staff</li>';
        echo '<li><strong>Security:</strong> 200 × 0.022 × 1.15 = 5 staff</li>';
        echo '<li><strong>Bartenders:</strong> min(200 × 0.023 × 1.20, 4) = 4 staff</li>';
        echo '<li><strong>Managers:</strong> 1 staff</li>';
        echo '</ul>';
        
    } else {
        echo '<p class="error">❌ Update failed: ' . $wpdb->last_error . '</p>';
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
}

echo '<hr>';
echo '<p><strong>🚀 Next Steps:</strong></p>';
echo '<ol>';
echo '<li>Test the manual planning with tables</li>';
echo '<li>Check if performance is improved</li>';
echo '<li>Verify wait staff and bottle girls increase with tables</li>';
echo '</ol>';
?>
