<?php
/**
 * Fix events list staffing display
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔧 Fix Events List Staffing</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

try {
    // 1. First, populate event features for all events
    echo '<h2>🏢 Populating Event Features</h2>';
    $table_features = $wpdb->prefix . 'coco_event_features';
    
    // Get all events
    $events = get_posts([
        'post_type' => 'tribe_events',
        'post_status' => 'publish',
        'numberposts' => -1
    ]);
    
    $features_updated = 0;
    foreach ($events as $event) {
        $event_id = $event->ID;
        
        // Check if features already exist
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_features WHERE event_id = %d",
            $event_id
        ));
        
        if (!$existing) {
            // Get event start date
            $start_date = get_post_meta($event_id, '_EventStartDate', true);
            $start_dt = $start_date ? new DateTime($start_date) : new DateTime();
            
            // Determine weekday
            $weekday = $start_dt->format('l');
            
            // Determine start bucket
            $hour = (int) $start_dt->format('H');
            if ($hour < 12) {
                $start_bucket = 'morning';
            } elseif ($hour < 17) {
                $start_bucket = 'afternoon';
            } elseif ($hour < 22) {
                $start_bucket = 'evening';
            } else {
                $start_bucket = 'late_night';
            }
            
            // Get venue capacity (default to 200 if not set - more realistic)
            $venue_id = get_post_meta($event_id, '_EventVenueID', true);
            $capacity = 200; // More realistic default capacity
            
            if ($venue_id) {
                $venue_capacity = get_post_meta($venue_id, '_venue_capacity', true);
                if ($venue_capacity) {
                    $capacity = (int) $venue_capacity;
                }
            }
            
            // Insert features
            $wpdb->insert(
                $table_features,
                [
                    'event_id' => $event_id,
                    'venue_id' => $venue_id,
                    'start_ts' => $start_dt->format('Y-m-d H:i:s'),
                    'weekday' => $weekday,
                    'start_bucket' => $start_bucket,
                    'capacity' => $capacity,
                    'tags_json' => '[]',
                    'holiday_flag' => 0,
                    'created_at' => current_time('mysql')
                ],
                ['%d', '%d', '%s', '%s', '%s', '%d', '%s', '%d', '%s']
            );
            
            $features_updated++;
        }
    }
    
    echo '<p class="success">✅ Updated ' . $features_updated . ' events with features</p>';
    
    // 2. Test the get_staffing_recommendations method directly
    echo '<h2>🧪 Testing Staffing Recommendations Method</h2>';
    
    // Get an event with planning data
    $table_planning = $wpdb->prefix . 'coco_event_planning';
    $planning_event = $wpdb->get_row("SELECT * FROM $table_planning WHERE est_attendance > 0 LIMIT 1");
    
    if ($planning_event) {
        $event_id = $planning_event->event_id;
        echo '<p>Testing with event ' . $event_id . ' (120 people, 4 tables)</p>';
        
        // Get the features for this event
        $features = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_features WHERE event_id = %d",
            $event_id
        ));
        
        echo '<h3>Event Features:</h3>';
        if ($features) {
            echo '<pre>' . print_r($features, true) . '</pre>';
        } else {
            echo '<p class="error">❌ No features found for event ' . $event_id . '</p>';
        }
        
        // Test the rules engine directly
        echo '<h3>Direct Rules Engine Test:</h3>';
        $rules_engine = new Coco_Ops_Rules_Engine();
        $result = $rules_engine->compute_staffing(
            $planning_event->est_attendance,
            [
                'weekday' => $features->weekday ?? 'Saturday',
                'start_bucket' => $features->start_bucket ?? 'evening'
            ],
            $planning_event->est_tables ?? 0
        );
        
        echo '<pre>' . print_r($result, true) . '</pre>';
        
        // Test the REST API method
        echo '<h3>REST API Method Test:</h3>';
        $rest_api = new Coco_Ops_REST_API();
        $reflection = new ReflectionClass($rest_api);
        $method = $reflection->getMethod('get_staffing_recommendations');
        $method->setAccessible(true);
        
        $staffing_result = $method->invoke($rest_api, $event_id, null, $planning_event, $features);
        echo '<pre>' . print_r($staffing_result, true) . '</pre>';
        
    } else {
        echo '<p class="error">❌ No planning data found to test with</p>';
    }
    
    // 3. Update the ruleset with table support
    echo '<h2>📋 Updating Ruleset</h2>';
    $table_rulesets = $wpdb->prefix . 'coco_rulesets';
    $current = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    
    if ($current) {
        $new_yaml = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022
    weekend_multiplier: 1.15
    weekday_min: 1
    weekend_min: 2
    max_count: 20
  bartenders:
    base_ratio: 0.023
    weekend_multiplier: 1.20
    min_count: 2
    max_count: 4
  wait_staff:
    base_ratio: 0.008
    weekend_multiplier: 1.50
    min_count: 1
    max_count: 15
    table_ratio: 0.5
  bottle_girls:
    base_ratio: 0.010
    weekend_multiplier: 1.15
    min_count: 0
    max_count: 10
    table_ratio: 0.3
  managers:
    base_ratio: 0.002
    weekend_multiplier: 1.00
    min_count: 1
    max_count: 1
    weekday_substitution: true";
        
        $result = $wpdb->update(
            $table_rulesets,
            ['yaml_content' => $new_yaml],
            ['id' => $current->id],
            ['%s'],
            ['%d']
        );
        
        if ($result !== false) {
            echo '<p class="success">✅ Ruleset updated with table support</p>';
        } else {
            echo '<p class="error">❌ Failed to update ruleset</p>';
        }
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p><strong>🚀 Next Steps:</strong></p>';
echo '<ol>';
echo '<li>Check the events page - capacity should now show 200 instead of 500</li>';
echo '<li>Staff rec should now show calculated numbers for events with planning</li>';
echo '<li>Tables should increase wait staff and bottle girls</li>';
echo '</ol>';
?>
