<?php
/**
 * Fix all issues: capacity, planning, sales, staffing
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔧 Fix All Issues</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
</style>';

global $wpdb;

try {
    // 1. Populate event features with capacity data
    echo '<h2>🏢 Populating Event Features</h2>';
    $table_features = $wpdb->prefix . 'coco_event_features';
    
    // Get all events
    $events = get_posts([
        'post_type' => 'tribe_events',
        'post_status' => 'publish',
        'numberposts' => -1
    ]);
    
    $features_updated = 0;
    foreach ($events as $event) {
        $event_id = $event->ID;
        
        // Check if features already exist
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_features WHERE event_id = %d",
            $event_id
        ));
        
        if (!$existing) {
            // Get event start date
            $start_date = get_post_meta($event_id, '_EventStartDate', true);
            $start_dt = $start_date ? new DateTime($start_date) : new DateTime();
            
            // Determine weekday
            $weekday = $start_dt->format('l');
            
            // Determine start bucket
            $hour = (int) $start_dt->format('H');
            if ($hour < 12) {
                $start_bucket = 'morning';
            } elseif ($hour < 17) {
                $start_bucket = 'afternoon';
            } elseif ($hour < 22) {
                $start_bucket = 'evening';
            } else {
                $start_bucket = 'late_night';
            }
            
            // Get venue capacity (default to 500 if not set)
            $venue_id = get_post_meta($event_id, '_EventVenueID', true);
            $capacity = 500; // Default capacity
            
            if ($venue_id) {
                $venue_capacity = get_post_meta($venue_id, '_venue_capacity', true);
                if ($venue_capacity) {
                    $capacity = (int) $venue_capacity;
                }
            }
            
            // Insert features
            $wpdb->insert(
                $table_features,
                [
                    'event_id' => $event_id,
                    'venue_id' => $venue_id,
                    'start_ts' => $start_dt->format('Y-m-d H:i:s'),
                    'weekday' => $weekday,
                    'start_bucket' => $start_bucket,
                    'capacity' => $capacity,
                    'tags_json' => '[]',
                    'holiday_flag' => 0,
                    'created_at' => current_time('mysql')
                ],
                ['%d', '%d', '%s', '%s', '%s', '%d', '%s', '%d', '%s']
            );
            
            $features_updated++;
        }
    }
    
    echo '<p class="success">✅ Updated ' . $features_updated . ' events with features</p>';
    
    // 2. Update ruleset with table support
    echo '<h2>📋 Updating Ruleset</h2>';
    $table_rulesets = $wpdb->prefix . 'coco_rulesets';
    $current = $wpdb->get_row("SELECT * FROM $table_rulesets WHERE active = 1 ORDER BY created_at DESC LIMIT 1");
    
    if ($current) {
        $new_yaml = "version: 1.0.0
rules:
  security:
    base_ratio: 0.022
    weekend_multiplier: 1.15
    weekday_min: 1
    weekend_min: 2
    max_count: 20
  bartenders:
    base_ratio: 0.023
    weekend_multiplier: 1.20
    min_count: 2
    max_count: 4
  wait_staff:
    base_ratio: 0.008
    weekend_multiplier: 1.50
    min_count: 1
    max_count: 15
    table_ratio: 0.5
  bottle_girls:
    base_ratio: 0.010
    weekend_multiplier: 1.15
    min_count: 0
    max_count: 10
    table_ratio: 0.3
  managers:
    base_ratio: 0.002
    weekend_multiplier: 1.00
    min_count: 1
    max_count: 1
    weekday_substitution: true";
        
        $result = $wpdb->update(
            $table_rulesets,
            ['yaml_content' => $new_yaml],
            ['id' => $current->id],
            ['%s'],
            ['%d']
        );
        
        if ($result !== false) {
            echo '<p class="success">✅ Ruleset updated with table support</p>';
        } else {
            echo '<p class="error">❌ Failed to update ruleset</p>';
        }
    }
    
    // 3. Test the complete flow
    echo '<h2>🧪 Testing Complete Flow</h2>';
    
    // Get an event with planning data
    $table_planning = $wpdb->prefix . 'coco_event_planning';
    $planning_event = $wpdb->get_row("SELECT * FROM $table_planning WHERE est_attendance > 0 LIMIT 1");
    
    if ($planning_event) {
        echo '<p>Testing with event ' . $planning_event->event_id . '</p>';
        
        // Test rules engine
        $rules_engine = new Coco_Ops_Rules_Engine();
        $result = $rules_engine->compute_staffing(
            $planning_event->est_attendance,
            ['weekday' => 'Saturday'],
            $planning_event->est_tables ?? 0
        );
        
        if (isset($result['staffing'])) {
            echo '<p class="success">✅ Rules engine working</p>';
            echo '<h3>Calculated Staffing:</h3>';
            echo '<ul>';
            foreach ($result['staffing'] as $role => $data) {
                echo '<li><strong>' . ucwords(str_replace('_', ' ', $role)) . ':</strong> ' . $data['count'] . '</li>';
            }
            echo '</ul>';
        } else {
            echo '<p class="error">❌ Rules engine failed</p>';
        }
    } else {
        echo '<p class="info">ℹ️ No planning data found to test with</p>';
    }
    
    // 4. Summary
    echo '<h2>📊 Summary</h2>';
    echo '<ul>';
    echo '<li><strong>Event Features:</strong> ' . $features_updated . ' events updated</li>';
    echo '<li><strong>Capacity:</strong> Should now show venue capacity</li>';
    echo '<li><strong>Staff Rec:</strong> Should calculate on-the-fly for events with planning</li>';
    echo '<li><strong>Tables:</strong> Now supported in calculations</li>';
    echo '</ul>';
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p><strong>🚀 Next Steps:</strong></p>';
echo '<ol>';
echo '<li>Go to Events page and check if capacity is now showing</li>';
echo '<li>Add planning data to an event (attendance + tables)</li>';
echo '<li>Check if staff rec shows calculated numbers</li>';
echo '<li>Test current sales data</li>';
echo '</ol>';
?>
