<?php
/**
 * Diagnostic Script for Historical Snapshots
 * Check system status and data availability
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Historical Snapshots Diagnostic</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    .warning { color: #F59E0B; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

echo '<h2>📊 System Status</h2>';

// Check if plugin tables exist
$tables_to_check = [
    $wpdb->prefix . 'coco_event_snapshots',
    $wpdb->prefix . 'coco_event_features',
    $wpdb->prefix . 'coco_staffing_recommendations'
];

foreach ($tables_to_check as $table) {
    $exists = $wpdb->get_var("SHOW TABLES LIKE '$table'");
    if ($exists) {
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table");
        echo '<p class="success">✅ Table ' . $table . ' exists (' . $count . ' rows)</p>';
    } else {
        echo '<p class="error">❌ Table ' . $table . ' does not exist</p>';
    }
}

echo '<h2>🎫 Events with Products</h2>';

// Check events with products linked - show actual date range
$events_with_products = $wpdb->get_results("
    SELECT p.ID, p.post_title, pm_start.meta_value as start_date, COUNT(pm_event.meta_value) as product_count
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
    LEFT JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
    WHERE p.post_type = 'tribe_events' 
    AND p.post_status = 'publish'
    AND pm_start.meta_value < NOW()
    GROUP BY p.ID
    ORDER BY pm_start.meta_value DESC
    LIMIT 20
");

// Get the actual date range
$date_range = $wpdb->get_row("
    SELECT 
        MIN(pm_start.meta_value) as earliest_date,
        MAX(pm_start.meta_value) as latest_date,
        COUNT(*) as total_events
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
    WHERE p.post_type = 'tribe_events' 
    AND p.post_status = 'publish'
    AND pm_start.meta_value < NOW()
");

echo '<p><strong>Actual date range of past events:</strong></p>';
echo '<ul>';
echo '<li><strong>Earliest event:</strong> ' . $date_range->earliest_date . '</li>';
echo '<li><strong>Latest event:</strong> ' . $date_range->latest_date . '</li>';
echo '<li><strong>Total past events:</strong> ' . $date_range->total_events . '</li>';
echo '</ul>';

echo '<p><strong>Recent past events and their product links:</strong></p>';
echo '<ul>';
foreach ($events_with_products as $event) {
    $start_date = new DateTime($event->start_date);
    $color = $event->product_count > 0 ? 'success' : 'error';
    echo '<li class="' . $color . '">';
    echo '<strong>' . $event->post_title . '</strong> - ' . $start_date->format('Y-m-d H:i');
    echo ' (' . $event->product_count . ' products)';
    echo '</li>';
}
echo '</ul>';

echo '<h2>🛒 WooCommerce Orders</h2>';

// Check total orders
$total_orders = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}wc_orders");
echo '<p><strong>Total WooCommerce orders:</strong> ' . $total_orders . '</p>';

// Check orders with event products
$orders_with_events = $wpdb->get_var("
    SELECT COUNT(DISTINCT wc_orders.id)
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        oim_ticket.meta_value = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    WHERE wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
");

echo '<p><strong>Orders with event products:</strong> ' . $orders_with_events . '</p>';

echo '<h2>📈 Test Query</h2>';

// Test the exact query we use
$test_orders = $wpdb->get_results("
    SELECT 
        p_ticket.post_title as ticket_name,
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total,
        pm_event.meta_value as event_id
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
    LIMIT 10
");

echo '<p><strong>Sample orders with event products:</strong></p>';
if ($test_orders) {
    echo '<ul>';
    foreach ($test_orders as $order) {
        echo '<li>';
        echo '<strong>' . $order->ticket_name . '</strong> - £' . $order->ticket_price;
        echo ' x' . $order->quantity . ' = £' . $order->line_total;
        echo ' (Event ID: ' . $order->event_id . ')';
        echo '</li>';
    }
    echo '</ul>';
} else {
    echo '<p class="error">❌ No orders found with event products</p>';
}

echo '<h2>🔧 Next Steps</h2>';

if ($orders_with_events == 0) {
    echo '<p class="error">❌ <strong>No orders found with event products!</strong></p>';
    echo '<p>This means either:</p>';
    echo '<ul>';
    echo '<li>Events don\'t have WooCommerce products linked</li>';
    echo '<li>No tickets have been sold for events</li>';
    echo '<li>The Events Calendar integration isn\'t set up properly</li>';
    echo '</ul>';
} else {
    echo '<p class="success">✅ <strong>Orders with event products found!</strong></p>';
    echo '<p>The issue might be with the specific events being processed or the date filtering.</p>';
}

echo '<hr>';
echo '<p>Diagnostic complete.</p>';
?>
