<?php
/**
 * Diagnose multiple issues
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Diagnose Issues</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

try {
    // 1. Check if events have planning data
    echo '<h2>📊 Events with Planning Data</h2>';
    $table_planning = $wpdb->prefix . 'coco_event_planning';
    $planning_events = $wpdb->get_results("SELECT * FROM $table_planning WHERE est_attendance > 0 LIMIT 5");
    
    if ($planning_events) {
        echo '<p class="success">✅ Found ' . count($planning_events) . ' events with planning data</p>';
        foreach ($planning_events as $plan) {
            echo '<p>Event ' . $plan->event_id . ': ' . $plan->est_attendance . ' people, ' . $plan->est_tables . ' tables</p>';
        }
    } else {
        echo '<p class="error">❌ No events with planning data found</p>';
    }
    
    // 2. Test rules engine calculation
    echo '<h2>🧮 Test Rules Engine</h2>';
    try {
        $rules_engine = new Coco_Ops_Rules_Engine();
        $test_result = $rules_engine->compute_staffing(200, ['weekday' => 'Saturday'], 10);
        
        if (isset($test_result['staffing'])) {
            echo '<p class="success">✅ Rules engine working</p>';
            echo '<pre>' . print_r($test_result['staffing'], true) . '</pre>';
        } else {
            echo '<p class="error">❌ Rules engine failed</p>';
        }
    } catch (Exception $e) {
        echo '<p class="error">❌ Rules engine error: ' . $e->getMessage() . '</p>';
    }
    
    // 3. Check capacity data source
    echo '<h2>🏢 Capacity Data Source</h2>';
    $table_features = $wpdb->prefix . 'coco_event_features';
    $features = $wpdb->get_results("SELECT event_id, capacity FROM $table_features WHERE capacity > 0 LIMIT 5");
    
    if ($features) {
        echo '<p class="success">✅ Found ' . count($features) . ' events with capacity data</p>';
        foreach ($features as $feature) {
            echo '<p>Event ' . $feature->event_id . ': Capacity ' . $feature->capacity . '</p>';
        }
    } else {
        echo '<p class="error">❌ No capacity data found</p>';
        echo '<p>Capacity should be set in event features. Check if events have features populated.</p>';
    }
    
    // 4. Test current sales endpoint
    echo '<h2>💰 Test Current Sales</h2>';
    $events_with_products = $wpdb->get_results("
        SELECT DISTINCT pm_event.meta_value as event_id
        FROM {$wpdb->postmeta} pm_event
        WHERE pm_event.meta_key = '_tribe_wooticket_for_event'
        LIMIT 3
    ");
    
    if ($events_with_products) {
        echo '<p class="success">✅ Found ' . count($events_with_products) . ' events with products</p>';
        
        foreach ($events_with_products as $event) {
            $event_id = $event->event_id;
            echo '<h3>Event ' . $event_id . ' Sales Test:</h3>';
            
            // Test the current sales query
            $orders = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    p_ticket.post_title as ticket_name,
                    pm_price.meta_value as ticket_price,
                    CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                    CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
                FROM {$wpdb->prefix}wc_orders wc_orders
                INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                    wc_orders.id = wc_order_items.order_id
                    AND wc_order_items.order_item_type = 'line_item'
                )
                INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                    wc_order_items.order_item_id = oim_ticket.order_item_id
                    AND oim_ticket.meta_key = '_product_id'
                )
                INNER JOIN {$wpdb->posts} p_ticket ON (
                    oim_ticket.meta_value = p_ticket.ID
                    AND p_ticket.post_type = 'product'
                )
                INNER JOIN {$wpdb->postmeta} pm_price ON (
                    p_ticket.ID = pm_price.post_id
                    AND pm_price.meta_key = '_price'
                )
                INNER JOIN {$wpdb->postmeta} pm_event ON (
                    p_ticket.ID = pm_event.post_id
                    AND pm_event.meta_key = '_tribe_wooticket_for_event'
                )
                LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                    wc_order_items.order_item_id = oim_qty.order_item_id
                    AND oim_qty.meta_key = '_qty'
                )
                LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                    wc_order_items.order_item_id = oim_line_total.order_item_id
                    AND oim_line_total.meta_key = '_line_total'
                )
                WHERE pm_event.meta_value = %d
                AND wc_orders.status IN ('wc-completed', 'wc-processing')
                AND wc_orders.status != 'wc-refunded'
                AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
                AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
            ", $event_id));
            
            echo '<p>Orders found: ' . count($orders) . '</p>';
            if (count($orders) > 0) {
                $tickets = 0;
                $tables = 0;
                $revenue = 0;
                $table_threshold = 100.00;
                
                foreach ($orders as $order) {
                    $price = (float) $order->ticket_price;
                    $qty = (int) $order->quantity;
                    $line_total = (float) $order->line_total;
                    
                    if ($price >= $table_threshold) {
                        $tables += $qty;
                    } else {
                        $tickets += $qty;
                    }
                    $revenue += $line_total;
                }
                
                echo '<p>Results: ' . $tickets . ' tickets, ' . $tables . ' tables, £' . $revenue . ' revenue</p>';
            }
        }
    } else {
        echo '<p class="error">❌ No events with products found</p>';
    }
    
    // 5. Test enrich_event_data method
    echo '<h2>🔧 Test Event Data Enrichment</h2>';
    if (!empty($planning_events)) {
        $test_event_id = $planning_events[0]->event_id;
        echo '<p>Testing with event ' . $test_event_id . '</p>';
        
        // Simulate the enrich_event_data call
        $rest_api = new Coco_Ops_REST_API();
        $reflection = new ReflectionClass($rest_api);
        $method = $reflection->getMethod('enrich_event_data');
        $method->setAccessible(true);
        
        $event_data = $method->invoke($rest_api, $test_event_id);
        
        if ($event_data) {
            echo '<p class="success">✅ Event data enrichment working</p>';
            echo '<h3>Staffing Data:</h3>';
            echo '<pre>' . print_r($event_data['staffing'], true) . '</pre>';
        } else {
            echo '<p class="error">❌ Event data enrichment failed</p>';
        }
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p>Diagnosis complete.</p>';
?>
