<?php
// Detailed cohort analysis for forecasting
require_once('../../../wp-config.php');

$event_id = 240721;

echo "🔍 DETAILED COHORT ANALYSIS FOR FORECASTING\n";
echo "Event ID: $event_id\n";
echo str_repeat("=", 80) . "\n\n";

global $wpdb;

// 1. Current event details
echo "1️⃣ CURRENT EVENT DETAILS\n";
echo str_repeat("-", 50) . "\n";
$event = get_post($event_id);
$event_start = get_post_meta($event_id, '_EventStartDate', true);
$venue_id = get_post_meta($event_id, '_EventVenueID', true);
$venue = get_post($venue_id);

echo "Event: {$event->post_title}\n";
echo "Date: $event_start\n";
echo "Venue: " . ($venue ? $venue->post_title : 'Unknown') . "\n\n";

// 2. Event features
$features = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features 
    WHERE event_id = %d
", $event_id));

echo "2️⃣ EVENT FEATURES\n";
echo str_repeat("-", 50) . "\n";
echo "Venue ID: {$features->venue_id}\n";
echo "Weekday: {$features->weekday}\n";
echo "Time Bucket: {$features->start_bucket}\n";
echo "Holiday Flag: {$features->holiday_flag}\n\n";

// 3. Cohort matching criteria
echo "3️⃣ COHORT MATCHING CRITERIA\n";
echo str_repeat("-", 50) . "\n";
echo "Matching events with:\n";
echo "  • Same venue: {$features->venue_id}\n";
echo "  • Same weekday: {$features->weekday}\n";
echo "  • Same time bucket: {$features->start_bucket}\n";
echo "  • Same holiday flag: {$features->holiday_flag}\n";
echo "  • Past events only: start_ts < NOW()\n";
echo "  • Excluding current event: $event_id\n\n";

// 4. Find cohort events using same logic as cohort matcher
$time_buckets = [];
switch ($features->start_bucket) {
    case 'late_night':
        $time_buckets = ['late_night', 'night'];
        break;
    case 'night':
        $time_buckets = ['night', 'late_night'];
        break;
    case 'evening':
        $time_buckets = ['evening', 'night'];
        break;
    case 'afternoon':
        $time_buckets = ['afternoon', 'evening'];
        break;
    case 'morning':
        $time_buckets = ['morning', 'afternoon'];
        break;
    default:
        $time_buckets = [$features->start_bucket];
}

$placeholders = implode(',', array_fill(0, count($time_buckets), '%s'));

$cohorts_query = "
    SELECT DISTINCT
        ef.*,
        COALESCE(ea.final_attendance, 0) as final_attendance,
        COALESCE(ea.final_revenue, 0) as final_revenue
    FROM {$wpdb->prefix}coco_event_features ef
    LEFT JOIN {$wpdb->prefix}coco_event_actuals ea ON ef.event_id = ea.event_id
    LEFT JOIN {$wpdb->prefix}coco_event_snapshots es ON ef.event_id = es.event_id
    WHERE ef.event_id != %d
    AND ef.venue_id = %s
    AND ef.weekday = %s
    AND ef.start_bucket IN ($placeholders)
    AND ef.holiday_flag = %d
    AND ef.start_ts < NOW()
    AND (ea.event_id IS NOT NULL OR es.event_id IS NOT NULL)
    ORDER BY ef.start_ts DESC
    LIMIT 20
";

$cohort_params = array_merge(
    [$event_id, $features->venue_id, $features->weekday],
    $time_buckets,
    [$features->holiday_flag]
);

$cohorts = $wpdb->get_results($wpdb->prepare($cohorts_query, $cohort_params));

echo "4️⃣ COHORT EVENTS FOUND\n";
echo str_repeat("-", 50) . "\n";
echo "Total cohorts: " . count($cohorts) . "\n\n";

foreach ($cohorts as $i => $cohort) {
    echo "Cohort " . ($i + 1) . ":\n";
    echo "  Event ID: {$cohort->event_id}\n";
    echo "  Date: {$cohort->start_ts}\n";
    echo "  Time Bucket: {$cohort->start_bucket}\n";
    
    // Get event title
    $event_title = $wpdb->get_var($wpdb->prepare("
        SELECT post_title FROM {$wpdb->posts} WHERE ID = %d
    ", $cohort->event_id));
    echo "  Title: $event_title\n";
    
    // Check actuals data
    $actuals = $wpdb->get_row($wpdb->prepare("
        SELECT * FROM {$wpdb->prefix}coco_event_actuals WHERE event_id = %d
    ", $cohort->event_id));
    
    if ($actuals) {
        echo "  ✅ Actuals Data:\n";
        echo "    Final Attendance: " . ($actuals->final_attendance ?: 'Not set') . "\n";
        echo "    Final Revenue: £" . ($actuals->final_revenue ?: 'Not set') . "\n";
        echo "    Entered At: {$actuals->entered_at}\n";
    } else {
        echo "  ❌ No Actuals Data\n";
    }
    
    // Check snapshots data
    $snapshots = $wpdb->get_results($wpdb->prepare("
        SELECT snapshot_ts, days_to_event, tickets_sold, tables_sold, revenue_to_date
        FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id = %d
        ORDER BY snapshot_ts DESC
        LIMIT 3
    ", $cohort->event_id));
    
    if ($snapshots) {
        echo "  📸 Snapshots Data:\n";
        foreach ($snapshots as $snapshot) {
            echo "    {$snapshot->snapshot_ts}: D{$snapshot->days_to_event}, {$snapshot->tickets_sold} tix, £{$snapshot->revenue_to_date}\n";
        }
    } else {
        echo "  ❌ No Snapshots Data\n";
    }
    
    echo "\n";
}

// 5. Data completeness analysis
echo "5️⃣ DATA COMPLETENESS ANALYSIS\n";
echo str_repeat("-", 50) . "\n";

$actuals_count = 0;
$snapshots_count = 0;
$complete_data_count = 0;

foreach ($cohorts as $cohort) {
    $has_actuals = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_actuals WHERE event_id = %d
    ", $cohort->event_id));
    
    $has_snapshots = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots WHERE event_id = %d
    ", $cohort->event_id));
    
    if ($has_actuals) $actuals_count++;
    if ($has_snapshots) $snapshots_count++;
    if ($has_actuals && $has_snapshots) $complete_data_count++;
}

echo "Total Cohort Events: " . count($cohorts) . "\n";
echo "Events with Actuals: $actuals_count\n";
echo "Events with Snapshots: $snapshots_count\n";
echo "Events with Both: $complete_data_count\n";
echo "Data Completeness: " . round(($complete_data_count / count($cohorts)) * 100, 1) . "%\n\n";

// 6. Walk-in analysis potential
echo "6️⃣ WALK-IN ANALYSIS POTENTIAL\n";
echo str_repeat("-", 50) . "\n";

$walk_in_analysis_events = [];
foreach ($cohorts as $cohort) {
    $actuals = $wpdb->get_row($wpdb->prepare("
        SELECT final_attendance FROM {$wpdb->prefix}coco_event_actuals WHERE event_id = %d
    ", $cohort->event_id));
    
    $final_snapshot = $wpdb->get_row($wpdb->prepare("
        SELECT tickets_sold FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id = %d
        ORDER BY snapshot_ts DESC
        LIMIT 1
    ", $cohort->event_id));
    
    if ($actuals && $actuals->final_attendance && $final_snapshot) {
        $walk_ins = $actuals->final_attendance - $final_snapshot->tickets_sold;
        $walk_in_analysis_events[] = [
            'event_id' => $cohort->event_id,
            'final_attendance' => $actuals->final_attendance,
            'final_tickets' => $final_snapshot->tickets_sold,
            'walk_ins' => $walk_ins
        ];
    }
}

echo "Events suitable for walk-in analysis: " . count($walk_in_analysis_events) . "\n\n";

if (count($walk_in_analysis_events) > 0) {
    echo "Walk-in Analysis Data:\n";
    foreach ($walk_in_analysis_events as $event_data) {
        echo "  Event {$event_data['event_id']}: {$event_data['final_attendance']} attendance - {$event_data['final_tickets']} tickets = {$event_data['walk_ins']} walk-ins\n";
    }
    
    // Calculate average walk-ins
    $total_walk_ins = array_sum(array_column($walk_in_analysis_events, 'walk_ins'));
    $avg_walk_ins = $total_walk_ins / count($walk_in_analysis_events);
    echo "\nAverage walk-ins: " . round($avg_walk_ins, 1) . "\n";
} else {
    echo "❌ No events have both attendance actuals and final ticket sales\n";
    echo "To enable walk-in analysis, you need to:\n";
    echo "  1. Add final attendance data to cohort events\n";
    echo "  2. Ensure final ticket sales are captured in snapshots\n";
}

echo "\n" . str_repeat("=", 80) . "\n";
echo "✅ DETAILED COHORT ANALYSIS COMPLETE\n";
?>
