<?php
/**
 * Debug script to test sales data retrieval
 */

// Include WordPress
require_once('../../../wp-config.php');

global $wpdb;

echo "<h2>🔍 Debug Sales Data Retrieval</h2>";

// Get the actual historical event ID from our query
$historical_event = $wpdb->get_row("
    SELECT p.ID, p.post_title, pm_start.meta_value as start_date
    FROM {$wpdb->posts} p
    INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
    INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
    WHERE p.post_type = 'tribe_events' 
    AND p.post_status = 'publish'
    AND pm_start.meta_value < NOW()
    ORDER BY pm_start.meta_value ASC
    LIMIT 1
");

if (!$historical_event) {
    echo "<p style='color: red;'>❌ No historical events found!</p>";
    exit;
}

$event_id = $historical_event->ID;
$event_start = new DateTime($historical_event->start_date);
$snapshot_date = clone $event_start;
$snapshot_date->modify('-1 day'); // D-1 (1 day before event)

echo "<h3>Testing Event: {$historical_event->post_title}</h3>";
echo "<p><strong>Event ID:</strong> {$event_id}</p>";
echo "<p><strong>Event Date:</strong> " . $event_start->format('Y-m-d H:i:s') . "</p>";
echo "<p><strong>Snapshot Date:</strong> " . $snapshot_date->format('Y-m-d H:i:s') . "</p>";

// First, check if the event has products linked
$products_linked = $wpdb->get_var($wpdb->prepare("
    SELECT COUNT(*)
    FROM {$wpdb->postmeta}
    WHERE meta_key = '_tribe_wooticket_for_event'
    AND meta_value = %d
", $event_id));

echo "<p><strong>Products linked to event:</strong> {$products_linked}</p>";

if ($products_linked == 0) {
    echo "<p style='color: red;'>❌ No products linked to this event!</p>";
    exit;
}

// Check if there are any orders for this event
$total_orders = $wpdb->get_var($wpdb->prepare("
    SELECT COUNT(DISTINCT wc_orders.id)
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        oim_ticket.meta_value = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
", $event_id));

echo "<p><strong>Total orders for this event:</strong> {$total_orders}</p>";

// Check orders up to snapshot date
$orders_up_to_date = $wpdb->get_var($wpdb->prepare("
    SELECT COUNT(DISTINCT wc_orders.id)
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        oim_ticket.meta_value = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND wc_orders.date_created_gmt <= %s
", $event_id, $snapshot_date->format('Y-m-d H:i:s')));

echo "<p><strong>Orders up to snapshot date:</strong> {$orders_up_to_date}</p>";

// Get sample order data
$sample_orders = $wpdb->get_results($wpdb->prepare("
    SELECT 
        wc_orders.id as order_id,
        wc_orders.date_created_gmt,
        wc_orders.status,
        p_ticket.post_title as ticket_name,
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND wc_orders.date_created_gmt <= %s
    LIMIT 5
", $event_id, $snapshot_date->format('Y-m-d H:i:s')));

echo "<h3>Sample Order Data:</h3>";
if (empty($sample_orders)) {
    echo "<p style='color: red;'>❌ No order data found!</p>";
} else {
    echo "<table border='1' style='border-collapse: collapse;'>";
    echo "<tr><th>Order ID</th><th>Date</th><th>Status</th><th>Ticket</th><th>Price</th><th>Qty</th><th>Total</th></tr>";
    foreach ($sample_orders as $order) {
        echo "<tr>";
        echo "<td>{$order->order_id}</td>";
        echo "<td>{$order->date_created_gmt}</td>";
        echo "<td>{$order->status}</td>";
        echo "<td>{$order->ticket_name}</td>";
        echo "<td>£{$order->ticket_price}</td>";
        echo "<td>{$order->quantity}</td>";
        echo "<td>£{$order->line_total}</td>";
        echo "</tr>";
    }
    echo "</table>";
}

echo "<h3>Debug Complete</h3>";
?>
