<?php
/**
 * Comprehensive Sales Debug for Event 243315
 */

// Include WordPress
require_once('../../../wp-config.php');

global $wpdb;

echo "🔍 Comprehensive Sales Debug for Event 243315\n";
echo "Time: " . current_time('mysql') . "\n\n";

$event_id = 243315;

// 1. Check if event exists and get details
$event = $wpdb->get_row($wpdb->prepare(
    "SELECT p.ID, p.post_title, pm_start.meta_value as start_date
     FROM {$wpdb->posts} p
     LEFT JOIN {$wpdb->postmeta} pm_start ON p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate'
     WHERE p.ID = %d",
    $event_id
));

if (!$event) {
    echo "❌ Event not found\n";
    exit;
}

echo "📅 Event: {$event->post_title}\n";
echo "📅 Start Date: {$event->start_date}\n\n";

// 2. Check what products are linked to this event
echo "🎫 Products linked to this event:\n";
$products = $wpdb->get_results($wpdb->prepare("
    SELECT p.ID, p.post_title, pm_event.meta_value as event_id
    FROM {$wpdb->posts} p
    JOIN {$wpdb->postmeta} pm_event ON p.ID = pm_event.post_id
    WHERE pm_event.meta_key = '_tribe_wooticket_for_event'
    AND pm_event.meta_value = %s
", $event_id));

foreach ($products as $product) {
    echo "  - Product {$product->ID}: {$product->post_title}\n";
}

if (empty($products)) {
    echo "  ❌ No products found linked to this event\n";
    echo "\n🔍 Let's check what meta keys exist for products:\n";
    
    $all_meta_keys = $wpdb->get_results("
        SELECT DISTINCT meta_key, COUNT(*) as count
        FROM {$wpdb->postmeta}
        WHERE meta_key LIKE '%event%' OR meta_key LIKE '%tribe%'
        GROUP BY meta_key
        ORDER BY count DESC
    ");
    
    foreach ($all_meta_keys as $meta) {
        echo "  - {$meta->meta_key}: {$meta->count} records\n";
    }
    
    echo "\n🔍 Let's check if event ID is stored differently:\n";
    $event_variations = $wpdb->get_results($wpdb->prepare("
        SELECT DISTINCT meta_key, meta_value, COUNT(*) as count
        FROM {$wpdb->postmeta}
        WHERE meta_value = %s
        GROUP BY meta_key, meta_value
        ORDER BY count DESC
    ", $event_id));
    
    foreach ($event_variations as $var) {
        echo "  - {$var->meta_key} = {$var->meta_value}: {$var->count} records\n";
    }
    
    exit;
}

echo "\n";

// 3. Check orders for these specific products
$product_ids = array_column($products, 'ID');
$placeholders = implode(',', array_fill(0, count($product_ids), '%d'));

echo "🛒 Orders for these products:\n";
$orders = $wpdb->get_results($wpdb->prepare("
    SELECT 
        oi.order_item_name,
        oi.order_item_type,
        oi.qty,
        o.date_created as order_date,
        o.total,
        o.status,
        oim_product.meta_value as product_id
    FROM {$wpdb->prefix}woocommerce_order_items oi
    JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_product ON oi.order_item_id = oim_product.order_item_id
    JOIN {$wpdb->prefix}wc_orders o ON oi.order_id = o.id
    WHERE oim_product.meta_key = '_product_id'
    AND oim_product.meta_value IN ($placeholders)
    AND o.status IN ('completed', 'processing')
    ORDER BY o.date_created ASC
", $product_ids));

echo "Found " . count($orders) . " orders\n";

$total_tickets = 0;
$total_tables = 0;
$total_revenue = 0;

foreach ($orders as $order) {
    echo "  - {$order->order_item_name}: {$order->qty} qty, £{$order->total} ({$order->order_date}) [Status: {$order->status}]\n";
    
    $total_tickets += $order->qty;
    $total_revenue += $order->total;
    
    if ($order->total >= 100) {
        $total_tables += $order->qty;
    }
}

echo "\n📊 Total Summary:\n";
echo "  - Total Tickets: {$total_tickets}\n";
echo "  - Total Tables: {$total_tables}\n";
echo "  - Total Revenue: £{$total_revenue}\n\n";

// 4. Check snapshots for comparison
echo "📸 Snapshots for comparison:\n";
$snapshots = $wpdb->get_results($wpdb->prepare(
    "SELECT * FROM {$wpdb->prefix}coco_event_snapshots WHERE event_id = %d ORDER BY snapshot_ts DESC LIMIT 3",
    $event_id
));

foreach ($snapshots as $snapshot) {
    echo "  - {$snapshot->snapshot_ts}: {$snapshot->tickets_sold} tickets, {$snapshot->tables_sold} tables, £{$snapshot->revenue_to_date}\n";
}

echo "\n🏁 Debug complete\n";
?>
