<?php
// Debug why ticket sales forecast is so low
require_once('../../../wp-config.php');

$event_id = 240721;

echo "🔍 DEBUGGING LOW TICKET SALES FORECAST\n";
echo "Event ID: $event_id\n";
echo str_repeat("=", 60) . "\n\n";

global $wpdb;

// 1. Check current sales vs forecast
echo "1️⃣ CURRENT SALES VS FORECAST\n";
echo str_repeat("-", 30) . "\n";

$live_sales = $wpdb->get_results($wpdb->prepare("
    SELECT 
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event_id));

$total_tickets = 0;
foreach ($live_sales as $sale) {
    $total_tickets += (int) $sale->quantity;
}

echo "Current Live Sales: $total_tickets tickets\n";

// 2. Check what snapshot data is being used
echo "\n2️⃣ SNAPSHOT DATA ANALYSIS\n";
echo str_repeat("-", 30) . "\n";

$snapshots = $wpdb->get_results($wpdb->prepare("
    SELECT snapshot_ts, days_to_event, tickets_sold, tables_sold, revenue_to_date
    FROM {$wpdb->prefix}coco_event_snapshots
    WHERE event_id = %d
    ORDER BY snapshot_ts DESC
", $event_id));

echo "Snapshots found: " . count($snapshots) . "\n";
if ($snapshots) {
    echo "Latest snapshot: {$snapshots[0]->snapshot_ts}\n";
    echo "Latest tickets: {$snapshots[0]->tickets_sold}\n";
    echo "Days to event: {$snapshots[0]->days_to_event}\n";
    
    // Check if snapshot is outdated
    $snapshot_time = strtotime($snapshots[0]->snapshot_ts);
    $event_time = strtotime('2025-10-25 22:30:00');
    $time_diff = $event_time - $snapshot_time;
    
    echo "Snapshot vs Event time difference: " . ($time_diff / 3600) . " hours\n";
    echo "Snapshot is " . ($time_diff > 0 ? "BEFORE event" : "AFTER event") . "\n";
}
echo "\n";

// 3. Check cohort data being used
echo "3️⃣ COHORT DATA ANALYSIS\n";
echo str_repeat("-", 30) . "\n";

$features = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features 
    WHERE event_id = %d
", $event_id));

$cohorts = $wpdb->get_results($wpdb->prepare("
    SELECT ef.event_id, ef.start_ts, p.post_title
    FROM {$wpdb->prefix}coco_event_features ef
    LEFT JOIN {$wpdb->posts} p ON ef.event_id = p.ID
    WHERE ef.venue_id = %d
    AND ef.weekday = %s
    AND ef.start_bucket IN ('late_night', 'night')
    AND ef.start_ts < NOW()
    AND ef.event_id != %d
    ORDER BY ef.start_ts DESC
    LIMIT 5
", $features->venue_id, $features->weekday, $event_id));

echo "Cohort events: " . count($cohorts) . "\n";
foreach ($cohorts as $cohort) {
    // Get final sales for this cohort
    $final_sales = $wpdb->get_var($wpdb->prepare("
        SELECT tickets_sold FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id = %d
        ORDER BY snapshot_ts DESC
        LIMIT 1
    ", $cohort->event_id));
    
    echo "  {$cohort->post_title}: $final_sales tickets\n";
}

// 4. Test the forecasting engine step by step
echo "\n4️⃣ FORECASTING ENGINE DEBUG\n";
echo str_repeat("-", 30) . "\n";

try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    
    // Use reflection to access private methods
    $reflection = new ReflectionClass($forecasting_engine);
    
    // Check if it's using live data or snapshots
    $get_live_sales_method = $reflection->getMethod('get_live_current_sales');
    $get_live_sales_method->setAccessible(true);
    $live_data = $get_live_sales_method->invoke($forecasting_engine, $event_id);
    
    echo "Live data check:\n";
    if ($live_data) {
        echo "  Using live data: " . json_encode($live_data) . "\n";
    } else {
        echo "  Not using live data (using snapshots)\n";
    }
    
    // Check current event status
    $event_start = get_post_meta($event_id, '_EventStartDate', true);
    $start_dt = new DateTime($event_start);
    $now_dt = new DateTime();
    $days_diff = $start_dt->diff($now_dt)->days;
    if ($start_dt < $now_dt) {
        $days_diff = -$days_diff;
    }
    
    echo "Event status: D$days_diff (event is " . ($days_diff < 0 ? "completed" : "upcoming") . ")\n";
    echo "Should use live data: " . (($days_diff >= -14 && $days_diff <= 14) ? "Yes" : "No") . "\n";
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Debug complete.\n";
?>
