<?php
/**
 * Debug HPOS sales query
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Debug HPOS Sales Query</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

try {
    // 1. Check if HPOS is enabled
    echo '<h2>🏪 WooCommerce HPOS Status</h2>';
    
    if (class_exists('Automattic\WooCommerce\Utilities\OrderUtil')) {
        $is_hpos_enabled = Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        echo '<p>HPOS Enabled: ' . ($is_hpos_enabled ? 'Yes' : 'No') . '</p>';
    } else {
        echo '<p class="info">ℹ️ OrderUtil class not available</p>';
    }
    
    // 2. Check if wc_orders table exists
    echo '<h2>📊 Database Tables Check</h2>';
    
    $wc_orders_table = $wpdb->prefix . 'wc_orders';
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$wc_orders_table'");
    
    if ($table_exists) {
        echo '<p class="success">✅ wc_orders table exists</p>';
        
        // Check if table has data
        $order_count = $wpdb->get_var("SELECT COUNT(*) FROM $wc_orders_table");
        echo '<p>Total orders in wc_orders: ' . $order_count . '</p>';
        
        // Check order statuses
        $statuses = $wpdb->get_results("
            SELECT status, COUNT(*) as count 
            FROM $wc_orders_table 
            GROUP BY status 
            ORDER BY count DESC
        ");
        
        echo '<h3>Order Statuses:</h3>';
        echo '<ul>';
        foreach ($statuses as $status) {
            echo '<li>' . $status->status . ': ' . $status->count . '</li>';
        }
        echo '</ul>';
        
    } else {
        echo '<p class="error">❌ wc_orders table does not exist</p>';
    }
    
    // 3. Check for events with products
    echo '<h2>🎫 Events with Products</h2>';
    
    $events_with_products = $wpdb->get_results("
        SELECT DISTINCT pm_event.meta_value as event_id, COUNT(*) as product_count
        FROM {$wpdb->postmeta} pm_event
        WHERE pm_event.meta_key = '_tribe_wooticket_for_event'
        GROUP BY pm_event.meta_value
        LIMIT 5
    ");
    
    if ($events_with_products) {
        echo '<p class="success">✅ Found ' . count($events_with_products) . ' events with products</p>';
        
        foreach ($events_with_products as $event) {
            echo '<p>Event ' . $event->event_id . ': ' . $event->product_count . ' products</p>';
        }
    } else {
        echo '<p class="error">❌ No events with products found</p>';
    }
    
    // 4. Test the HPOS query step by step
    if (!empty($events_with_products)) {
        $test_event_id = $events_with_products[0]->event_id;
        echo '<h2>🧪 Test HPOS Query for Event ' . $test_event_id . '</h2>';
        
        // Step 1: Check if event has products
        $product_count = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*)
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_tribe_wooticket_for_event'
            AND meta_value = %d
        ", $test_event_id));
        
        echo '<p>Products linked to event: ' . $product_count . '</p>';
        
        if ($product_count > 0) {
            // Step 2: Get the products
            $products = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID, p.post_title, pm_price.meta_value as price
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->postmeta} pm_event ON (
                    p.ID = pm_event.post_id
                    AND pm_event.meta_key = '_tribe_wooticket_for_event'
                )
                INNER JOIN {$wpdb->postmeta} pm_price ON (
                    p.ID = pm_price.post_id
                    AND pm_price.meta_key = '_price'
                )
                WHERE pm_event.meta_value = %d
            ", $test_event_id));
            
            echo '<h3>Products for this event:</h3>';
            foreach ($products as $product) {
                echo '<p>- ' . $product->post_title . ' (ID: ' . $product->ID . ', Price: £' . $product->price . ')</p>';
            }
            
            // Step 3: Test the full HPOS query
            echo '<h3>Full HPOS Query Test:</h3>';
            
            $orders = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    p_ticket.post_title as ticket_name,
                    pm_price.meta_value as ticket_price,
                    CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                    CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total,
                    wc_orders.status as order_status
                FROM {$wpdb->prefix}wc_orders wc_orders
                INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                    wc_orders.id = wc_order_items.order_id
                    AND wc_order_items.order_item_type = 'line_item'
                )
                INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                    wc_order_items.order_item_id = oim_ticket.order_item_id
                    AND oim_ticket.meta_key = '_product_id'
                )
                INNER JOIN {$wpdb->posts} p_ticket ON (
                    oim_ticket.meta_value = p_ticket.ID
                    AND p_ticket.post_type = 'product'
                )
                INNER JOIN {$wpdb->postmeta} pm_price ON (
                    p_ticket.ID = pm_price.post_id
                    AND pm_price.meta_key = '_price'
                )
                INNER JOIN {$wpdb->postmeta} pm_event ON (
                    p_ticket.ID = pm_event.post_id
                    AND pm_event.meta_key = '_tribe_wooticket_for_event'
                )
                LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                    wc_order_items.order_item_id = oim_qty.order_item_id
                    AND oim_qty.meta_key = '_qty'
                )
                LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                    wc_order_items.order_item_id = oim_line_total.order_item_id
                    AND oim_line_total.meta_key = '_line_total'
                )
                WHERE pm_event.meta_value = %d
                AND wc_orders.status IN ('wc-completed', 'wc-processing')
                AND wc_orders.status != 'wc-refunded'
                AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
                AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
            ", $test_event_id));
            
            echo '<p>Orders found: ' . count($orders) . '</p>';
            
            if (count($orders) > 0) {
                echo '<h3>Order Details:</h3>';
                $tickets = 0;
                $tables = 0;
                $revenue = 0;
                $table_threshold = 100.00;
                
                foreach ($orders as $order) {
                    $price = (float) $order->ticket_price;
                    $qty = (int) $order->quantity;
                    $line_total = (float) $order->line_total;
                    
                    echo '<p>' . $order->ticket_name . ' - £' . $price . ' x' . $qty . ' = £' . $line_total . ' (Status: ' . $order->order_status . ')</p>';
                    
                    if ($price >= $table_threshold) {
                        $tables += $qty;
                    } else {
                        $tickets += $qty;
                    }
                    $revenue += $line_total;
                }
                
                echo '<p class="success">✅ Results: ' . $tickets . ' tickets, ' . $tables . ' tables, £' . $revenue . ' revenue</p>';
            } else {
                echo '<p class="error">❌ No orders found</p>';
                
                // Check if there are any orders at all
                $any_orders = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}wc_orders");
                echo '<p>Total orders in system: ' . $any_orders . '</p>';
                
                // Check if there are orders for this event but with different status
                $all_orders = $wpdb->get_results($wpdb->prepare("
                    SELECT wc_orders.status, COUNT(*) as count
                    FROM {$wpdb->prefix}wc_orders wc_orders
                    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                        wc_orders.id = wc_order_items.order_id
                        AND wc_order_items.order_item_type = 'line_item'
                    )
                    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                        wc_order_items.order_item_id = oim_ticket.order_item_id
                        AND oim_ticket.meta_key = '_product_id'
                    )
                    INNER JOIN {$wpdb->postmeta} pm_event ON (
                        oim_ticket.meta_value = pm_event.post_id
                        AND pm_event.meta_key = '_tribe_wooticket_for_event'
                    )
                    WHERE pm_event.meta_value = %d
                    GROUP BY wc_orders.status
                ", $test_event_id));
                
                if ($all_orders) {
                    echo '<h3>Orders for this event (all statuses):</h3>';
                    foreach ($all_orders as $order) {
                        echo '<p>' . $order->status . ': ' . $order->count . '</p>';
                    }
                }
            }
        }
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p>HPOS debug complete.</p>';
?>
