<?php
// Debug forecast sources and snapshot timing
require_once('../../../wp-config.php');

$event_id = 250474;

echo "🔍 Debugging Forecast Sources for Event $event_id\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// 1. Check all snapshots for this event
echo "1️⃣ Event Snapshots:\n";
$snapshots = $wpdb->get_results($wpdb->prepare("
    SELECT snapshot_ts, days_to_event, tickets_sold, tables_sold, revenue_to_date
    FROM {$wpdb->prefix}coco_event_snapshots
    WHERE event_id = %d
    ORDER BY snapshot_ts DESC
", $event_id));

echo "Found " . count($snapshots) . " snapshots:\n";
foreach ($snapshots as $snapshot) {
    echo "- {$snapshot->snapshot_ts}: D{$snapshot->days_to_event}, {$snapshot->tickets_sold} tix, {$snapshot->tables_sold} tables, £{$snapshot->revenue_to_date}\n";
}

// 2. Check which snapshot is being used for forecasting
echo "\n2️⃣ Snapshot Used for Forecasting:\n";
$event_start = get_post_meta($event_id, '_EventStartDate', true);
$now = new DateTime();
$start_dt = new DateTime($event_start);
$days_to_event = $start_dt->diff($now)->days;
if ($start_dt < $now) {
    $days_to_event = -$days_to_event; // Event is in the past
}

echo "Event start: $event_start\n";
echo "Current time: " . $now->format('Y-m-d H:i:s') . "\n";
echo "Days to event: $days_to_event\n";

// Check if it's a current event (within 14 days)
$is_current_event = ($days_to_event >= -14 && $days_to_event <= 14);
echo "Is current event: " . ($is_current_event ? 'Yes' : 'No') . "\n";

if ($is_current_event) {
    echo "Using live sales data instead of snapshots\n";
} else {
    echo "Using snapshot data for forecasting\n";
}

// 3. Check live sales data
echo "\n3️⃣ Live Sales Data:\n";
$live_sales = $wpdb->get_results($wpdb->prepare("
    SELECT 
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event_id));

$total_tickets = 0;
foreach ($live_sales as $sale) {
    $total_tickets += (int) $sale->quantity;
}

echo "Live sales tickets: $total_tickets\n";
echo "Live sales records: " . count($live_sales) . "\n";

// 4. Check cohorts and their final sales
echo "\n4️⃣ Cohort Analysis:\n";
$event_features = $wpdb->get_row($wpdb->prepare("
    SELECT venue_id, weekday, time_bucket
    FROM {$wpdb->prefix}coco_event_features
    WHERE event_id = %d
", $event_id));

if ($event_features) {
    $cohorts = $wpdb->get_results($wpdb->prepare("
        SELECT ef.event_id, ef.start_ts, p.post_title
        FROM {$wpdb->prefix}coco_event_features ef
        LEFT JOIN {$wpdb->posts} p ON ef.event_id = p.ID
        WHERE ef.venue_id = %d
        AND ef.weekday = %s
        AND ef.time_bucket = %s
        AND ef.event_id != %d
        ORDER BY ef.start_ts DESC
        LIMIT 10
    ", $event_features->venue_id, $event_features->weekday, $event_features->time_bucket, $event_id));
    
    echo "Found " . count($cohorts) . " cohort events\n";
    
    $cohort_final_sales = [];
    foreach ($cohorts as $cohort) {
        $final_snapshot = $wpdb->get_row($wpdb->prepare("
            SELECT tickets_sold, snapshot_ts
            FROM {$wpdb->prefix}coco_event_snapshots
            WHERE event_id = %d
            ORDER BY snapshot_ts DESC
            LIMIT 1
        ", $cohort->event_id));
        
        if ($final_snapshot) {
            $cohort_final_sales[] = $final_snapshot->tickets_sold;
            echo "- Event {$cohort->event_id}: {$final_snapshot->tickets_sold} tickets (snapshot: {$final_snapshot->snapshot_ts})\n";
        } else {
            echo "- Event {$cohort->event_id}: No snapshot data\n";
        }
    }
    
    if (!empty($cohort_final_sales)) {
        sort($cohort_final_sales);
        $p50 = $cohort_final_sales[floor(count($cohort_final_sales) * 0.50)];
        echo "\nCohort final sales p50: $p50\n";
        echo "This is likely where the 40 comes from!\n";
    }
}

// 5. Test the forecasting engine step by step
echo "\n5️⃣ Forecasting Engine Debug:\n";
try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    
    // Use reflection to access private methods
    $reflection = new ReflectionClass($forecasting_engine);
    
    // Check if it's using live data or snapshots
    $get_live_sales_method = $reflection->getMethod('get_live_current_sales');
    $get_live_sales_method->setAccessible(true);
    $live_data = $get_live_sales_method->invoke($forecasting_engine, $event_id);
    
    if ($live_data) {
        echo "Using live sales data: " . json_encode($live_data) . "\n";
    } else {
        echo "Not using live sales data\n";
    }
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Debug complete.\n";
?>
