<?php
// Debug forecast cohorts for event 250474
require_once('../../../wp-config.php');

$event_id = 250474; // Cite - Afrobeats Saturdays at Cococure Cite

echo "🔍 Debugging Forecast Cohorts for Event $event_id\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// Get event details
$event = $wpdb->get_row($wpdb->prepare("
    SELECT p.post_title, p.post_date, pm.meta_value as start_ts
    FROM {$wpdb->posts} p
    LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_EventStartDate'
    WHERE p.ID = %d
", $event_id));

if ($event) {
    echo "📅 Event Details:\n";
    echo "Title: " . $event->post_title . "\n";
    echo "Event Date: " . $event->start_ts . "\n";
    echo "Days to event: " . round((strtotime($event->start_ts) - time()) / 86400, 1) . "\n\n";
}

// Get current sales data
$current_sales = $wpdb->get_row($wpdb->prepare("
    SELECT tickets_sold, tables_sold, revenue_to_date
    FROM {$wpdb->prefix}coco_event_snapshots
    WHERE event_id = %d
    ORDER BY snapshot_ts DESC
    LIMIT 1
", $event_id));

echo "📊 Current Sales (from snapshot):\n";
echo "Tickets: " . ($current_sales->tickets_sold ?? 'N/A') . "\n";
echo "Tables: " . ($current_sales->tables_sold ?? 'N/A') . "\n";
echo "Revenue: £" . ($current_sales->revenue_to_date ?? 'N/A') . "\n\n";

// Get live sales breakdown
$live_sales = $wpdb->get_results($wpdb->prepare("
    SELECT 
        oi.meta_value as ticket_price,
        oi2.meta_value as quantity,
        oi3.meta_value as line_total
    FROM {$wpdb->prefix}wc_orders o
    JOIN {$wpdb->prefix}wc_orders_meta oi ON o.id = oi.order_id AND oi.meta_key = '_tribe_wooticket_for_event'
    JOIN {$wpdb->prefix}wc_orders_meta oi2 ON o.id = oi2.order_id AND oi2.meta_key = '_qty'
    JOIN {$wpdb->prefix}wc_orders_meta oi3 ON o.id = oi3.order_id AND oi3.meta_key = '_line_total'
    WHERE oi.meta_value = %s
    AND o.status IN ('wc-completed', 'wc-processing')
", $event_id));

$paid_tickets = 0;
$free_tickets = 0;
$table_entries = 0;
$total_tickets = 0;
$table_threshold = 100;

echo "💰 Live Sales Breakdown:\n";
foreach ($live_sales as $sale) {
    $price = (float) $sale->ticket_price;
    $qty = (int) $sale->quantity;
    $line_total = (float) $sale->line_total;
    
    $total_tickets += $qty;
    
    if ($price >= $table_threshold) {
        $table_entries += $qty * 6;
        echo "Table: £$price x$qty = £$line_total → {$qty} tables = " . ($qty * 6) . " entries\n";
    } elseif ($line_total > 0) {
        $paid_tickets += $qty;
        echo "Paid: £$price x$qty = £$line_total\n";
    } else {
        $free_tickets += $qty;
        echo "Free: £$price x$qty = £$line_total\n";
    }
}

echo "\n📈 Summary:\n";
echo "Total tickets: $total_tickets\n";
echo "Paid tickets: $paid_tickets\n";
echo "Free tickets: $free_tickets\n";
echo "Table entries: $table_entries\n";
echo "Paid + table entries: " . ($paid_tickets + $table_entries) . "\n\n";

// Get event features
$event_features = $wpdb->get_row($wpdb->prepare("
    SELECT venue_id, weekday, time_bucket, start_ts
    FROM {$wpdb->prefix}coco_event_features
    WHERE event_id = %d
", $event_id));

if ($event_features) {
    echo "🏢 Event Features:\n";
    echo "Venue ID: " . $event_features->venue_id . "\n";
    echo "Weekday: " . $event_features->weekday . "\n";
    echo "Time Bucket: " . $event_features->time_bucket . "\n";
    echo "Start TS: " . $event_features->start_ts . "\n\n";
} else {
    echo "❌ No event features found!\n\n";
}

// Find cohorts manually
echo "🔍 Finding Cohorts:\n";
if ($event_features) {
    $cohorts = $wpdb->get_results($wpdb->prepare("
        SELECT ef.event_id, ef.weekday, ef.time_bucket, ef.start_ts,
               ef.venue_id, p.post_title
        FROM {$wpdb->prefix}coco_event_features ef
        LEFT JOIN {$wpdb->posts} p ON ef.event_id = p.ID
        WHERE ef.venue_id = %d
        AND ef.weekday = %s
        AND ef.time_bucket = %s
        AND ef.event_id != %d
        ORDER BY ef.start_ts DESC
        LIMIT 20
    ", $event_features->venue_id, $event_features->weekday, $event_features->time_bucket, $event_id));
    
    echo "Found " . count($cohorts) . " cohort events:\n";
    foreach ($cohorts as $cohort) {
        echo "- Event {$cohort->event_id}: {$cohort->post_title} ({$cohort->start_ts})\n";
    }
    echo "\n";
    
    // Get final sales for each cohort
    echo "📊 Cohort Final Sales:\n";
    $cohort_final_sales = [];
    foreach ($cohorts as $cohort) {
        // Get final snapshot (most recent)
        $final_snapshot = $wpdb->get_row($wpdb->prepare("
            SELECT tickets_sold, tables_sold, revenue_to_date, snapshot_ts
            FROM {$wpdb->prefix}coco_event_snapshots
            WHERE event_id = %d
            ORDER BY snapshot_ts DESC
            LIMIT 1
        ", $cohort->event_id));
        
        if ($final_snapshot) {
            $cohort_final_sales[] = $final_snapshot->tickets_sold;
            echo "- Event {$cohort->event_id}: {$final_snapshot->tickets_sold} tickets (snapshot: {$final_snapshot->snapshot_ts})\n";
        } else {
            echo "- Event {$cohort->event_id}: No snapshot data\n";
        }
    }
    
    if (!empty($cohort_final_sales)) {
        sort($cohort_final_sales);
        $p25 = $cohort_final_sales[floor(count($cohort_final_sales) * 0.25)];
        $p50 = $cohort_final_sales[floor(count($cohort_final_sales) * 0.50)];
        $p75 = $cohort_final_sales[floor(count($cohort_final_sales) * 0.75)];
        
        echo "\n📈 Cohort Percentiles:\n";
        echo "P25: $p25\n";
        echo "P50: $p50\n";
        echo "P75: $p75\n";
        echo "Count: " . count($cohort_final_sales) . "\n";
        
        echo "\n🔍 Rule Check:\n";
        $minimum = $paid_tickets + $table_entries;
        echo "Minimum (paid + tables): $minimum\n";
        echo "Cohort P50: $p50\n";
        echo "Max(forecast, minimum): " . max($p50, $minimum) . "\n";
    }
}

// Test the actual forecasting engine
echo "\n🤖 Testing Actual Forecasting Engine:\n";
try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    $forecast = $forecasting_engine->generate_forecast($event_id);
    
    if ($forecast && isset($forecast['forecast'])) {
        echo "Generated forecast:\n";
        echo "- Ticket sales p50: " . ($forecast['forecast']['ticket_sales_p50'] ?? 'N/A') . "\n";
        echo "- Ticket sales p25: " . ($forecast['forecast']['ticket_sales_p25'] ?? 'N/A') . "\n";
        echo "- Ticket sales p75: " . ($forecast['forecast']['ticket_sales_p75'] ?? 'N/A') . "\n";
        echo "- Attendance p50: " . ($forecast['forecast']['attendance_p50'] ?? 'N/A') . "\n";
        echo "- Revenue p50: £" . ($forecast['forecast']['revenue_p50'] ?? 'N/A') . "\n";
        echo "- Confidence: " . ($forecast['forecast']['confidence'] ?? 'N/A') . "\n";
        echo "- Cohort count: " . ($forecast['forecast']['cohort_count'] ?? 'N/A') . "\n";
    } else {
        echo "❌ Failed to generate forecast\n";
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Debug complete.\n";
?>
