<?php
// Debug specific issues with event 250474
require_once('../../../wp-config.php');

$event_id = 250474;

echo "🔍 Debugging Event 250474 Issues\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// 1. Check if event exists
echo "1️⃣ Event Existence Check:\n";
$event = $wpdb->get_row($wpdb->prepare("
    SELECT ID, post_title, post_status, post_type
    FROM {$wpdb->posts}
    WHERE ID = %d
", $event_id));

if ($event) {
    echo "✅ Event exists: {$event->post_title} ({$event->post_status})\n";
} else {
    echo "❌ Event not found!\n";
    exit;
}

// 2. Check event features
echo "\n2️⃣ Event Features Check:\n";
$features = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features
    WHERE event_id = %d
", $event_id));

if ($features) {
    echo "✅ Event features found:\n";
    foreach ($features as $key => $value) {
        echo "  $key: $value\n";
    }
} else {
    echo "❌ No event features - this is the problem!\n";
    echo "Event needs to be in coco_event_features table for forecasting\n";
}

// 3. Check snapshots
echo "\n3️⃣ Snapshot Data Check:\n";
$snapshots = $wpdb->get_results($wpdb->prepare("
    SELECT snapshot_ts, days_to_event, tickets_sold, tables_sold, revenue_to_date
    FROM {$wpdb->prefix}coco_event_snapshots
    WHERE event_id = %d
    ORDER BY snapshot_ts DESC
    LIMIT 5
", $event_id));

echo "Found " . count($snapshots) . " snapshots:\n";
foreach ($snapshots as $snapshot) {
    echo "- {$snapshot->snapshot_ts}: D{$snapshot->days_to_event}, {$snapshot->tickets_sold} tix, {$snapshot->tables_sold} tables, £{$snapshot->revenue_to_date}\n";
}

// 4. Check live sales data with different approaches
echo "\n4️⃣ Live Sales Data Check:\n";

// Approach 1: Direct WooCommerce query
$wc_orders = $wpdb->get_results($wpdb->prepare("
    SELECT 
        oi.meta_value as ticket_price,
        oi2.meta_value as quantity,
        oi3.meta_value as line_total,
        o.status
    FROM {$wpdb->prefix}wc_orders o
    JOIN {$wpdb->prefix}wc_orders_meta oi ON o.id = oi.order_id AND oi.meta_key = '_tribe_wooticket_for_event'
    JOIN {$wpdb->prefix}wc_orders_meta oi2 ON o.id = oi2.order_id AND oi2.meta_key = '_qty'
    JOIN {$wpdb->prefix}wc_orders_meta oi3 ON o.id = oi3.order_id AND oi3.meta_key = '_line_total'
    WHERE oi.meta_value = %s
    LIMIT 10
", $event_id));

echo "WooCommerce orders found: " . count($wc_orders) . "\n";
foreach ($wc_orders as $order) {
    echo "- Price: £{$order->ticket_price}, Qty: {$order->quantity}, Total: £{$order->line_total}, Status: {$order->status}\n";
}

// Approach 2: Check if event has products
echo "\n5️⃣ Event Products Check:\n";
$products = $wpdb->get_results($wpdb->prepare("
    SELECT p.ID, p.post_title, pm.meta_value as event_id
    FROM {$wpdb->posts} p
    JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_tribe_wooticket_for_event'
    WHERE pm.meta_value = %s
    AND p.post_type = 'product'
", $event_id));

echo "Products linked to event: " . count($products) . "\n";
foreach ($products as $product) {
    echo "- Product {$product->ID}: {$product->post_title}\n";
}

// Approach 3: Check order statuses
echo "\n6️⃣ Order Status Check:\n";
$statuses = $wpdb->get_results($wpdb->prepare("
    SELECT o.status, COUNT(*) as count
    FROM {$wpdb->prefix}wc_orders o
    JOIN {$wpdb->prefix}wc_orders_meta oi ON o.id = oi.order_id AND oi.meta_key = '_tribe_wooticket_for_event'
    WHERE oi.meta_value = %s
    GROUP BY o.status
", $event_id));

foreach ($statuses as $status) {
    echo "- {$status->status}: {$status->count} orders\n";
}

// 7. Check if we need to populate event features
echo "\n7️⃣ Event Features Population:\n";
if (!$features) {
    echo "Event needs to be added to coco_event_features table\n";
    echo "This is required for forecasting to work\n";
    
    // Try to populate it
    $start_date = get_post_meta($event_id, '_EventStartDate', true);
    $venue_id = get_post_meta($event_id, '_EventVenueID', true);
    
    if ($start_date && $venue_id) {
        $start_ts = new DateTime($start_date);
        $weekday = strtolower($start_ts->format('l'));
        
        // Determine time bucket
        $hour = (int) $start_ts->format('H');
        if ($hour < 12) {
            $time_bucket = 'morning';
        } elseif ($hour < 17) {
            $time_bucket = 'afternoon';
        } elseif ($hour < 21) {
            $time_bucket = 'evening';
        } elseif ($hour < 23) {
            $time_bucket = 'night';
        } else {
            $time_bucket = 'late_night';
        }
        
        echo "Would populate with:\n";
        echo "- Venue ID: $venue_id\n";
        echo "- Weekday: $weekday\n";
        echo "- Time Bucket: $time_bucket\n";
        echo "- Start TS: $start_date\n";
    }
}

echo "\n✅ Debug complete.\n";
?>
