<?php
// Debug compute-staffing endpoint
require_once('../../../wp-config.php');

$event_id = 252283;

echo "🔍 Debugging Compute-Staffing for Event $event_id\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// 1. Check if event exists
echo "1️⃣ Event Check:\n";
$event = get_post($event_id);
if ($event) {
    echo "✅ Event found: {$event->post_title}\n";
} else {
    echo "❌ Event not found\n";
    exit;
}

// 2. Check event features
echo "\n2️⃣ Event Features Check:\n";
$features = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features 
    WHERE event_id = %d
", $event_id));

if ($features) {
    echo "✅ Event features found:\n";
    echo "- Venue ID: {$features->venue_id}\n";
    echo "- Weekday: {$features->weekday}\n";
    echo "- Time bucket: {$features->start_bucket}\n";
} else {
    echo "❌ No event features found\n";
}

// 3. Check active ruleset
echo "\n3️⃣ Ruleset Check:\n";
$ruleset = $wpdb->get_row("
    SELECT * FROM {$wpdb->prefix}coco_staffing_rulesets 
    WHERE is_active = 1 
    ORDER BY created_at DESC 
    LIMIT 1
");

if ($ruleset) {
    echo "✅ Active ruleset found:\n";
    echo "- ID: {$ruleset->id}\n";
    echo "- Name: {$ruleset->name}\n";
    echo "- Created: {$ruleset->created_at}\n";
    echo "- YAML length: " . strlen($ruleset->yaml_content) . " chars\n";
} else {
    echo "❌ No active ruleset found\n";
}

// 4. Test rules engine
echo "\n4️⃣ Rules Engine Test:\n";
try {
    require_once('includes/class-rules-engine.php');
    $rules_engine = new Coco_Ops_Rules_Engine();
    
    // Test with sample data
    $attendance = 100;
    $tables = 5;
    $event_features = [
        'weekday' => $features->weekday ?? 'saturday',
        'start_bucket' => $features->start_bucket ?? 'night'
    ];
    
    echo "Testing with: attendance=$attendance, tables=$tables\n";
    echo "Event features: " . json_encode($event_features) . "\n";
    
    $result = $rules_engine->compute_staffing($attendance, $event_features, $tables);
    
    if (isset($result['error'])) {
        echo "❌ Rules engine error: " . $result['error'] . "\n";
    } else {
        echo "✅ Rules engine working:\n";
        echo "Staffing: " . json_encode($result['staffing']) . "\n";
        if (isset($result['rationale'])) {
            echo "Rationale: " . $result['rationale'] . "\n";
        }
    }
    
} catch (Exception $e) {
    echo "❌ Exception: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
} catch (Error $e) {
    echo "❌ Fatal Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

// 5. Test forecasting engine
echo "\n5️⃣ Forecasting Engine Test:\n";
try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    $forecast = $forecasting_engine->generate_forecast($event_id);
    
    if ($forecast && isset($forecast['forecast'])) {
        echo "✅ Forecast generated:\n";
        echo "- Attendance p50: " . ($forecast['forecast']['attendance_p50'] ?? 'N/A') . "\n";
        echo "- Confidence: " . ($forecast['forecast']['confidence'] ?? 'N/A') . "\n";
    } else {
        echo "❌ No forecast available\n";
    }
    
} catch (Exception $e) {
    echo "❌ Forecast exception: " . $e->getMessage() . "\n";
}

echo "\n✅ Debug complete.\n";
?>
