<?php
// Debug cohort matching issue
require_once('../../../wp-config.php');

$event_id = 240721;

echo "🔍 DEBUGGING COHORT MATCHING ISSUE\n";
echo "Event ID: $event_id\n";
echo str_repeat("=", 60) . "\n\n";

global $wpdb;

// 1. Check current event features
echo "1️⃣ CURRENT EVENT FEATURES\n";
echo str_repeat("-", 30) . "\n";
$current_event = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features 
    WHERE event_id = %d
", $event_id));

if ($current_event) {
    echo "Event ID: {$current_event->event_id}\n";
    echo "Venue ID: {$current_event->venue_id}\n";
    echo "Weekday: {$current_event->weekday}\n";
    echo "Time Bucket: {$current_event->start_bucket}\n";
    echo "Start Time: {$current_event->start_ts}\n";
    echo "Holiday Flag: {$current_event->holiday_flag}\n";
} else {
    echo "❌ No event features found!\n";
    exit;
}
echo "\n";

// 2. Check what the cohort query should return
echo "2️⃣ COHORT QUERY TEST\n";
echo str_repeat("-", 30) . "\n";

$where_conditions = ["ef.event_id != %d"];
$query_params = [$event_id];

$where_conditions[] = "ef.venue_id = %s";
$query_params[] = $current_event->venue_id;

$where_conditions[] = "ef.weekday = %s";
$query_params[] = $current_event->weekday;

// Time bucket matching - late_night and night are the same
$time_buckets = [];
switch ($current_event->start_bucket) {
    case 'late_night':
        $time_buckets = ['late_night', 'night'];
        break;
    case 'night':
        $time_buckets = ['night', 'late_night'];
        break;
    case 'evening':
        $time_buckets = ['evening', 'night'];
        break;
    case 'afternoon':
        $time_buckets = ['afternoon', 'evening'];
        break;
    case 'morning':
        $time_buckets = ['morning', 'afternoon'];
        break;
    default:
        $time_buckets = [$current_event->start_bucket];
}

$placeholders = implode(',', array_fill(0, count($time_buckets), '%s'));
$where_conditions[] = "ef.start_bucket IN ($placeholders)";
$query_params = array_merge($query_params, $time_buckets);

$where_conditions[] = "ef.holiday_flag = %d";
$query_params[] = $current_event->holiday_flag;

// CRITICAL: Only past events
$where_conditions[] = "ef.start_ts < NOW()";

$where_clause = implode(' AND ', $where_conditions);

$query = "
    SELECT DISTINCT
        ef.*,
        COALESCE(ea.final_attendance, 0) as final_attendance,
        COALESCE(ea.final_revenue, 0) as final_revenue
    FROM {$wpdb->prefix}coco_event_features ef
    LEFT JOIN {$wpdb->prefix}coco_event_actuals ea ON ef.event_id = ea.event_id
    LEFT JOIN {$wpdb->prefix}coco_event_snapshots es ON ef.event_id = es.event_id
    WHERE $where_clause
    ORDER BY ef.start_ts DESC
    LIMIT 20
";

echo "Query: " . str_replace(["\n", "  "], [" ", " "], $query) . "\n";
echo "Params: " . implode(', ', $query_params) . "\n\n";

$cohorts = $wpdb->get_results($wpdb->prepare($query, $query_params));

echo "Cohorts found: " . count($cohorts) . "\n\n";

if (count($cohorts) > 0) {
    echo "Cohort Events:\n";
    foreach ($cohorts as $i => $cohort) {
        $is_past = strtotime($cohort->start_ts) < time();
        $status = $is_past ? "✅ PAST" : "❌ FUTURE";
        echo "  " . ($i + 1) . ". {$cohort->start_ts} - {$status}\n";
    }
} else {
    echo "❌ No cohorts found!\n";
}
echo "\n";

// 3. Check if there are any past events with same criteria
echo "3️⃣ CHECKING FOR PAST EVENTS\n";
echo str_repeat("-", 30) . "\n";

$past_query = "
    SELECT ef.event_id, ef.start_ts, p.post_title
    FROM {$wpdb->prefix}coco_event_features ef
    LEFT JOIN {$wpdb->posts} p ON ef.event_id = p.ID
    WHERE ef.venue_id = %s
    AND ef.weekday = %s
    AND ef.start_bucket IN ($placeholders)
    AND ef.holiday_flag = %d
    AND ef.start_ts < NOW()
    AND ef.event_id != %d
    ORDER BY ef.start_ts DESC
    LIMIT 10
";

$past_params = array_merge(
    [$current_event->venue_id, $current_event->weekday],
    $time_buckets,
    [$current_event->holiday_flag, $event_id]
);

$past_events = $wpdb->get_results($wpdb->prepare($past_query, $past_params));

echo "Past events found: " . count($past_events) . "\n";
foreach ($past_events as $event) {
    echo "  • {$event->post_title} ({$event->start_ts})\n";
}
echo "\n";

// 4. Check current time vs event time
echo "4️⃣ TIME COMPARISON\n";
echo str_repeat("-", 30) . "\n";
echo "Current time: " . date('Y-m-d H:i:s') . "\n";
echo "Event time: {$current_event->start_ts}\n";
echo "Event is in past: " . (strtotime($current_event->start_ts) < time() ? 'Yes' : 'No') . "\n";

echo "\n✅ Debug complete.\n";
?>
