<?php
/**
 * Debug auto forecast issues
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Debug Auto Forecast Issues</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

try {
    // 1. Test current sales endpoint
    echo '<h2>💰 Test Current Sales Endpoint</h2>';
    
    // Get an event with products
    $events_with_products = $wpdb->get_results("
        SELECT DISTINCT pm_event.meta_value as event_id
        FROM {$wpdb->postmeta} pm_event
        WHERE pm_event.meta_key = '_tribe_wooticket_for_event'
        LIMIT 3
    ");
    
    if ($events_with_products) {
        foreach ($events_with_products as $event) {
            $event_id = $event->event_id;
            echo '<h3>Event ' . $event_id . ' Sales Test:</h3>';
            
            // Test the current sales query directly
            $orders = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    p_ticket.post_title as ticket_name,
                    pm_price.meta_value as ticket_price,
                    CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
                    CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
                FROM {$wpdb->prefix}wc_orders wc_orders
                INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
                    wc_orders.id = wc_order_items.order_id
                    AND wc_order_items.order_item_type = 'line_item'
                )
                INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
                    wc_order_items.order_item_id = oim_ticket.order_item_id
                    AND oim_ticket.meta_key = '_product_id'
                )
                INNER JOIN {$wpdb->posts} p_ticket ON (
                    oim_ticket.meta_value = p_ticket.ID
                    AND p_ticket.post_type = 'product'
                )
                INNER JOIN {$wpdb->postmeta} pm_price ON (
                    p_ticket.ID = pm_price.post_id
                    AND pm_price.meta_key = '_price'
                )
                INNER JOIN {$wpdb->postmeta} pm_event ON (
                    p_ticket.ID = pm_event.post_id
                    AND pm_event.meta_key = '_tribe_wooticket_for_event'
                )
                LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
                    wc_order_items.order_item_id = oim_qty.order_item_id
                    AND oim_qty.meta_key = '_qty'
                )
                LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
                    wc_order_items.order_item_id = oim_line_total.order_item_id
                    AND oim_line_total.meta_key = '_line_total'
                )
                WHERE pm_event.meta_value = %d
                AND wc_orders.status IN ('wc-completed', 'wc-processing')
                AND wc_orders.status != 'wc-refunded'
                AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
                AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
            ", $event_id));
            
            echo '<p>Orders found: ' . count($orders) . '</p>';
            
            if (count($orders) > 0) {
                $tickets = 0;
                $tables = 0;
                $revenue = 0;
                $table_threshold = 100.00;
                
                foreach ($orders as $order) {
                    $price = (float) $order->ticket_price;
                    $qty = (int) $order->quantity;
                    $line_total = (float) $order->line_total;
                    
                    echo '<p>Order: ' . $order->ticket_name . ' - £' . $price . ' x' . $qty . ' = £' . $line_total . '</p>';
                    
                    if ($price >= $table_threshold) {
                        $tables += $qty;
                    } else {
                        $tickets += $qty;
                    }
                    $revenue += $line_total;
                }
                
                echo '<p class="success">✅ Results: ' . $tickets . ' tickets, ' . $tables . ' tables, £' . $revenue . ' revenue</p>';
            } else {
                echo '<p class="error">❌ No orders found</p>';
            }
        }
    } else {
        echo '<p class="error">❌ No events with products found</p>';
    }
    
    // 2. Test the REST API endpoint
    echo '<h2>🌐 Test REST API Endpoint</h2>';
    
    if (!empty($events_with_products)) {
        $test_event_id = $events_with_products[0]->event_id;
        echo '<p>Testing REST API for event ' . $test_event_id . '</p>';
        
        $rest_api = new Coco_Ops_REST_API();
        $mock_request = new stdClass();
        $mock_request->get_param = function($param) use ($test_event_id) {
            return $param === 'id' ? $test_event_id : null;
        };
        
        $response = $rest_api->get_current_sales($mock_request);
        
        if ($response && !is_wp_error($response)) {
            echo '<p class="success">✅ REST API working</p>';
            echo '<pre>' . print_r($response->data, true) . '</pre>';
        } else {
            echo '<p class="error">❌ REST API failed</p>';
            if ($response) {
                echo '<pre>' . print_r($response, true) . '</pre>';
            }
        }
    }
    
    // 3. Test forecasting engine
    echo '<h2>🔮 Test Forecasting Engine</h2>';
    
    if (!empty($events_with_products)) {
        $test_event_id = $events_with_products[0]->event_id;
        echo '<p>Testing forecasting engine for event ' . $test_event_id . '</p>';
        
        try {
            $forecasting_engine = new Coco_Ops_Forecasting_Engine();
            $forecast = $forecasting_engine->generate_forecast($test_event_id);
            
            if ($forecast) {
                echo '<p class="success">✅ Forecasting engine working</p>';
                echo '<pre>' . print_r($forecast, true) . '</pre>';
            } else {
                echo '<p class="error">❌ Forecasting engine returned null</p>';
            }
        } catch (Exception $e) {
            echo '<p class="error">❌ Forecasting engine error: ' . $e->getMessage() . '</p>';
            echo '<pre>' . $e->getTraceAsString() . '</pre>';
        }
    }
    
    // 4. Test compute staffing with forecast
    echo '<h2>🧮 Test Compute Staffing with Forecast</h2>';
    
    if (!empty($events_with_products)) {
        $test_event_id = $events_with_products[0]->event_id;
        echo '<p>Testing compute staffing for event ' . $test_event_id . '</p>';
        
        // Create a mock request for compute_staffing
        $mock_request = new stdClass();
        $mock_request->get_param = function($param) use ($test_event_id) {
            switch ($param) {
                case 'id': return $test_event_id;
                case 'use_forecast': return true;
                default: return null;
            }
        };
        
        $rest_api = new Coco_Ops_REST_API();
        $response = $rest_api->compute_staffing($mock_request);
        
        if ($response && !is_wp_error($response)) {
            echo '<p class="success">✅ Compute staffing working</p>';
            echo '<pre>' . print_r($response->data, true) . '</pre>';
        } else {
            echo '<p class="error">❌ Compute staffing failed</p>';
            if ($response) {
                echo '<pre>' . print_r($response, true) . '</pre>';
            }
        }
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p>Debug complete.</p>';
?>
