<?php
// Comprehensive forecast analysis for event 240721
require_once('../../../wp-config.php');

$event_id = 240721;

echo "🔍 COMPREHENSIVE FORECAST ANALYSIS FOR EVENT $event_id\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n";
echo str_repeat("=", 80) . "\n\n";

global $wpdb;

// 1. EVENT BASIC INFO
echo "1️⃣ EVENT BASIC INFORMATION\n";
echo str_repeat("-", 40) . "\n";
$event = get_post($event_id);
if ($event) {
    echo "Event Title: {$event->post_title}\n";
    echo "Event ID: $event_id\n";
    echo "Post Status: {$event->post_status}\n";
} else {
    echo "❌ Event not found!\n";
    exit;
}

$event_start = get_post_meta($event_id, '_EventStartDate', true);
$event_end = get_post_meta($event_id, '_EventEndDate', true);
$venue_id = get_post_meta($event_id, '_EventVenueID', true);

echo "Event Start: $event_start\n";
echo "Event End: $event_end\n";
echo "Venue ID: $venue_id\n";

// Calculate days to event
$start_dt = new DateTime($event_start);
$now_dt = new DateTime();
$days_diff = $start_dt->diff($now_dt)->days;
if ($start_dt < $now_dt) {
    $days_diff = -$days_diff; // Event is in the past
}
echo "Days to Event: $days_diff\n";

// Get venue name
$venue = get_post($venue_id);
$venue_name = $venue ? $venue->post_title : 'Unknown';
echo "Venue Name: $venue_name\n\n";

// 2. EVENT FEATURES
echo "2️⃣ EVENT FEATURES\n";
echo str_repeat("-", 40) . "\n";
$features = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features 
    WHERE event_id = %d
", $event_id));

if ($features) {
    echo "✅ Event features found:\n";
    echo "Venue ID: {$features->venue_id}\n";
    echo "Weekday: {$features->weekday}\n";
    echo "Time Bucket: {$features->start_bucket}\n";
    echo "Start Time: {$features->start_time}\n";
    echo "Created: {$features->created_at}\n";
} else {
    echo "❌ No event features found!\n";
    exit;
}
echo "\n";

// 3. CURRENT SALES DATA
echo "3️⃣ CURRENT SALES DATA\n";
echo str_repeat("-", 40) . "\n";

// Get live sales data
$live_sales = $wpdb->get_results($wpdb->prepare("
    SELECT 
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event_id));

$total_tickets = 0;
$total_tables = 0;
$total_revenue = 0;
$paid_tickets = 0;
$free_tickets = 0;
$table_entries = 0;

echo "Live Sales Analysis:\n";
echo "Total sales records: " . count($live_sales) . "\n\n";

foreach ($live_sales as $sale) {
    $price = (float) $sale->ticket_price;
    $qty = (int) $sale->quantity;
    $total = (float) $sale->line_total;
    
    $total_tickets += $qty;
    $total_revenue += $total;
    
    if ($price >= 100) {
        $total_tables += $qty;
        $table_entries += $qty * 6; // Assuming 6 people per table
        echo "Table: £$price x$qty = £$total → {$qty} tables = " . ($qty * 6) . " entries\n";
    } elseif ($price > 0) {
        $paid_tickets += $qty;
        echo "Paid ticket: £$price x$qty = £$total\n";
    } else {
        $free_tickets += $qty;
        echo "Free ticket: £$price x$qty = £$total\n";
    }
}

echo "\nSummary:\n";
echo "Total tickets sold: $total_tickets\n";
echo "Paid tickets: $paid_tickets\n";
echo "Free tickets: $free_tickets\n";
echo "Tables sold: $total_tables\n";
echo "Table entries: $table_entries\n";
echo "Total revenue: £$total_revenue\n";
echo "Paid tickets + table entries: " . ($paid_tickets + $table_entries) . "\n\n";

// 4. SNAPSHOTS DATA
echo "4️⃣ SNAPSHOTS DATA\n";
echo str_repeat("-", 40) . "\n";
$snapshots = $wpdb->get_results($wpdb->prepare("
    SELECT snapshot_ts, days_to_event, tickets_sold, tables_sold, revenue_to_date
    FROM {$wpdb->prefix}coco_event_snapshots
    WHERE event_id = %d
    ORDER BY snapshot_ts DESC
", $event_id));

echo "Snapshots found: " . count($snapshots) . "\n";
if ($snapshots) {
    echo "Latest snapshot: {$snapshots[0]->snapshot_ts} (D{$snapshots[0]->days_to_event})\n";
    echo "Latest tickets: {$snapshots[0]->tickets_sold}\n";
    echo "Latest tables: {$snapshots[0]->tables_sold}\n";
    echo "Latest revenue: £{$snapshots[0]->revenue_to_date}\n";
    
    // Check for D+1 snapshot
    $d1_snapshot = $wpdb->get_row($wpdb->prepare("
        SELECT snapshot_ts, days_to_event, tickets_sold, tables_sold, revenue_to_date
        FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id = %d
        AND days_to_event < 0
        AND days_to_event >= -1
        ORDER BY snapshot_ts DESC
        LIMIT 1
    ", $event_id));
    
    if ($d1_snapshot) {
        echo "\nD+1 Snapshot (Final Sales):\n";
        echo "Date: {$d1_snapshot->snapshot_ts}\n";
        echo "Days: D{$d1_snapshot->days_to_event}\n";
        echo "Final tickets: {$d1_snapshot->tickets_sold}\n";
        echo "Final tables: {$d1_snapshot->tables_sold}\n";
        echo "Final revenue: £{$d1_snapshot->revenue_to_date}\n";
    } else {
        echo "\n❌ No D+1 snapshot found\n";
    }
}
echo "\n";

// 5. COHORT ANALYSIS
echo "5️⃣ COHORT ANALYSIS\n";
echo str_repeat("-", 40) . "\n";

// Find cohort events
$cohorts = $wpdb->get_results($wpdb->prepare("
    SELECT ef.event_id, ef.start_ts, p.post_title, ef.venue_id, ef.weekday, ef.start_bucket
    FROM {$wpdb->prefix}coco_event_features ef
    LEFT JOIN {$wpdb->posts} p ON ef.event_id = p.ID
    WHERE ef.venue_id = %d
    AND ef.weekday = %s
    AND ef.start_bucket = %s
    AND ef.event_id != %d
    ORDER BY ef.start_ts DESC
    LIMIT 20
", $features->venue_id, $features->weekday, $features->start_bucket, $event_id));

echo "Cohort Criteria:\n";
echo "- Venue ID: {$features->venue_id}\n";
echo "- Weekday: {$features->weekday}\n";
echo "- Time Bucket: {$features->start_bucket}\n";
echo "- Excluding current event: $event_id\n\n";

echo "Cohort Events Found: " . count($cohorts) . "\n\n";

$cohort_ids = array_column($cohorts, 'event_id');
$cohort_final_sales = [];

if (!empty($cohort_ids)) {
    // Get final sales for cohorts (prioritizing D+1)
    $placeholders = implode(',', array_fill(0, count($cohort_ids), '%d'));
    
    $d1_query = "
        SELECT event_id, tickets_sold, snapshot_ts, days_to_event
        FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id IN ($placeholders)
        AND days_to_event < 0
        AND days_to_event >= -1
        AND snapshot_ts = (
            SELECT MAX(snapshot_ts) 
            FROM {$wpdb->prefix}coco_event_snapshots s2 
            WHERE s2.event_id = {$wpdb->prefix}coco_event_snapshots.event_id
            AND s2.days_to_event < 0
            AND s2.days_to_event >= -1
        )
    ";
    
    $d1_results = $wpdb->get_results($wpdb->prepare($d1_query, $cohort_ids));
    $d1_sales = [];
    foreach ($d1_results as $result) {
        $d1_sales[$result->event_id] = $result->tickets_sold;
    }
    
    // Get fallback for events without D+1
    $events_with_d1 = array_keys($d1_sales);
    $events_without_d1 = array_diff($cohort_ids, $events_with_d1);
    
    $fallback_sales = [];
    if (!empty($events_without_d1)) {
        $placeholders_fallback = implode(',', array_fill(0, count($events_without_d1), '%d'));
        $fallback_query = "
            SELECT event_id, tickets_sold, snapshot_ts, days_to_event
            FROM {$wpdb->prefix}coco_event_snapshots
            WHERE event_id IN ($placeholders_fallback)
            AND snapshot_ts = (
                SELECT MAX(snapshot_ts) 
                FROM {$wpdb->prefix}coco_event_snapshots s2 
                WHERE s2.event_id = {$wpdb->prefix}coco_event_snapshots.event_id
            )
        ";
        
        $fallback_results = $wpdb->get_results($wpdb->prepare($fallback_query, $events_without_d1));
        foreach ($fallback_results as $result) {
            $fallback_sales[$result->event_id] = $result->tickets_sold;
        }
    }
    
    $cohort_final_sales = array_merge($d1_sales, $fallback_sales);
    
    echo "Cohort Final Sales Data:\n";
    echo "Events with D+1 snapshots: " . count($events_with_d1) . "\n";
    echo "Events with fallback snapshots: " . count($events_without_d1) . "\n\n";
    
    foreach ($cohorts as $cohort) {
        $final_tickets = $cohort_final_sales[$cohort->event_id] ?? 'No data';
        $has_d1 = in_array($cohort->event_id, $events_with_d1);
        $source = $has_d1 ? 'D+1' : 'Latest';
        
        echo "Event {$cohort->event_id}: {$cohort->post_title}\n";
        echo "  Date: {$cohort->start_ts}\n";
        echo "  Final tickets: $final_tickets (Source: $source)\n";
        echo "\n";
    }
    
    // Calculate percentiles
    $final_sales_values = array_values($cohort_final_sales);
    sort($final_sales_values);
    
    $p25 = $final_sales_values[floor(count($final_sales_values) * 0.25)];
    $p50 = $final_sales_values[floor(count($final_sales_values) * 0.50)];
    $p75 = $final_sales_values[floor(count($final_sales_values) * 0.75)];
    
    echo "Cohort Final Sales Percentiles:\n";
    echo "P25: $p25 tickets\n";
    echo "P50: $p50 tickets\n";
    echo "P75: $p75 tickets\n\n";
}

// 6. PACING ANALYSIS
echo "6️⃣ PACING ANALYSIS\n";
echo str_repeat("-", 40) . "\n";

if ($snapshots && !empty($cohort_final_sales)) {
    $current_snapshot = $snapshots[0];
    $current_tickets = $current_snapshot->tickets_sold;
    $current_days = $current_snapshot->days_to_event;
    
    echo "Current Snapshot:\n";
    echo "Date: {$current_snapshot->snapshot_ts}\n";
    echo "Days to event: D$current_days\n";
    echo "Tickets sold: $current_tickets\n\n";
    
    // Get cohort snapshots at similar days
    $cohort_snapshots_at_day = $wpdb->get_results($wpdb->prepare("
        SELECT event_id, tickets_sold, days_to_event
        FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id IN ($placeholders)
        AND ABS(days_to_event - %f) <= 1
        ORDER BY ABS(days_to_event - %f)
    ", array_merge($cohort_ids, [$current_days, $current_days])));
    
    echo "Cohort Snapshots at Similar Days (D$current_days ±1):\n";
    echo "Found " . count($cohort_snapshots_at_day) . " snapshots\n\n";
    
    $pacing_ratios = [];
    foreach ($cohort_snapshots_at_day as $cohort_snapshot) {
        if (isset($cohort_final_sales[$cohort_snapshot->event_id])) {
            $cohort_tickets_at_day = $cohort_snapshot->tickets_sold;
            $cohort_final_tickets = $cohort_final_sales[$cohort_snapshot->event_id];
            
            if ($cohort_tickets_at_day > 0) {
                $ratio = $cohort_final_tickets / $cohort_tickets_at_day;
                $pacing_ratios[] = $ratio;
                
                echo "Event {$cohort_snapshot->event_id}: {$cohort_tickets_at_day} → {$cohort_final_tickets} (ratio: " . number_format($ratio, 2) . "x)\n";
            }
        }
    }
    
    if (!empty($pacing_ratios)) {
        sort($pacing_ratios);
        $p50_ratio = $pacing_ratios[floor(count($pacing_ratios) * 0.50)];
        $p25_ratio = $pacing_ratios[floor(count($pacing_ratios) * 0.25)];
        $p75_ratio = $pacing_ratios[floor(count($pacing_ratios) * 0.75)];
        
        echo "\nPacing Ratios:\n";
        echo "P25: " . number_format($p25_ratio, 2) . "x\n";
        echo "P50: " . number_format($p50_ratio, 2) . "x\n";
        echo "P75: " . number_format($p75_ratio, 2) . "x\n\n";
        
        // Calculate pacing forecast
        $pacing_ticket_p50 = (int) ($current_tickets * $p50_ratio);
        $pacing_ticket_p25 = (int) ($current_tickets * $p25_ratio);
        $pacing_ticket_p75 = (int) ($current_tickets * $p75_ratio);
        
        echo "Pacing-Based Ticket Sales Forecast:\n";
        echo "P25: $current_tickets × " . number_format($p25_ratio, 2) . " = $pacing_ticket_p25 tickets\n";
        echo "P50: $current_tickets × " . number_format($p50_ratio, 2) . " = $pacing_ticket_p50 tickets\n";
        echo "P75: $current_tickets × " . number_format($p75_ratio, 2) . " = $pacing_ticket_p75 tickets\n\n";
    } else {
        echo "❌ No pacing ratios calculated (insufficient data)\n\n";
    }
}

// 7. ACTUAL FORECASTING ENGINE TEST
echo "7️⃣ ACTUAL FORECASTING ENGINE TEST\n";
echo str_repeat("-", 40) . "\n";

try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    $forecast = $forecasting_engine->generate_forecast($event_id);
    
    if ($forecast && isset($forecast['forecast'])) {
        echo "✅ Forecast Generated Successfully!\n\n";
        
        echo "Ticket Sales Forecast:\n";
        echo "P25: " . ($forecast['forecast']['ticket_sales_p25'] ?? 'N/A') . " tickets\n";
        echo "P50: " . ($forecast['forecast']['ticket_sales_p50'] ?? 'N/A') . " tickets\n";
        echo "P75: " . ($forecast['forecast']['ticket_sales_p75'] ?? 'N/A') . " tickets\n\n";
        
        echo "Attendance Forecast:\n";
        echo "P25: " . ($forecast['forecast']['attendance_p25'] ?? 'N/A') . " attendees\n";
        echo "P50: " . ($forecast['forecast']['attendance_p50'] ?? 'N/A') . " attendees\n";
        echo "P75: " . ($forecast['forecast']['attendance_p75'] ?? 'N/A') . " attendees\n\n";
        
        echo "Revenue Forecast:\n";
        echo "P50: £" . ($forecast['forecast']['revenue_p50'] ?? 'N/A') . "\n\n";
        
        echo "Forecast Metadata:\n";
        echo "Confidence: " . ($forecast['forecast']['confidence'] ?? 'N/A') . "\n";
        echo "Method: " . ($forecast['forecast']['method'] ?? 'N/A') . "\n";
        echo "Cohort Count: " . ($forecast['forecast']['cohort_count'] ?? 'N/A') . "\n";
        
        if (isset($forecast['current'])) {
            echo "\nCurrent Data Used:\n";
            echo "Tickets sold: " . ($forecast['current']['tickets_sold'] ?? 'N/A') . "\n";
            echo "Tables sold: " . ($forecast['current']['tables_sold'] ?? 'N/A') . "\n";
            echo "Revenue: £" . ($forecast['current']['revenue_to_date'] ?? 'N/A') . "\n";
            echo "Days to event: " . ($forecast['current']['days_to_event'] ?? 'N/A') . "\n";
        }
        
        if (isset($forecast['pacing'])) {
            echo "\nPacing Analysis:\n";
            echo "Status: " . ($forecast['pacing']['status'] ?? 'N/A') . "\n";
            if (isset($forecast['pacing']['ratio'])) {
                echo "P50 Ratio: " . number_format($forecast['pacing']['ratio'], 2) . "x\n";
            }
        }
        
    } else {
        echo "❌ Failed to generate forecast\n";
        if (isset($forecast['error'])) {
            echo "Error: " . $forecast['error'] . "\n";
        }
    }
    
} catch (Exception $e) {
    echo "❌ Exception: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}

echo "\n" . str_repeat("=", 80) . "\n";
echo "✅ COMPREHENSIVE ANALYSIS COMPLETE\n";
?>
