<?php
// Comprehensive fix and test for event 250474
require_once('../../../wp-config.php');

$event_id = 250474;

echo "🔧 Comprehensive Fix and Test for Event $event_id\n";
echo "Date: " . date('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// Step 1: Fix event features
echo "1️⃣ Fixing Event Features:\n";
$start_date = get_post_meta($event_id, '_EventStartDate', true);
$venue_id = get_post_meta($event_id, '_EventVenueID', true);

if ($start_date && $venue_id) {
    $start_ts = new DateTime($start_date);
    $weekday = strtolower($start_ts->format('l'));
    
    $hour = (int) $start_ts->format('H');
    if ($hour < 12) {
        $time_bucket = 'morning';
    } elseif ($hour < 17) {
        $time_bucket = 'afternoon';
    } elseif ($hour < 21) {
        $time_bucket = 'evening';
    } elseif ($hour < 23) {
        $time_bucket = 'night';
    } else {
        $time_bucket = 'late_night';
    }
    
    // Check if already exists
    $existing = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_features WHERE event_id = %d
    ", $event_id));
    
    if ($existing == 0) {
        $result = $wpdb->insert(
            $wpdb->prefix . 'coco_event_features',
            [
                'event_id' => $event_id,
                'venue_id' => $venue_id,
                'weekday' => $weekday,
                'time_bucket' => $time_bucket,
                'start_ts' => $start_date,
                'created_at' => current_time('mysql')
            ],
            ['%d', '%d', '%s', '%s', '%s', '%s']
        );
        
        if ($result) {
            echo "✅ Added event features\n";
        } else {
            echo "❌ Failed to add event features: " . $wpdb->last_error . "\n";
        }
    } else {
        echo "✅ Event features already exist\n";
    }
} else {
    echo "❌ Missing event data\n";
}

// Step 2: Test sales query with different statuses
echo "\n2️⃣ Testing Sales Query:\n";
$statuses_to_try = [
    ['wc-completed', 'wc-processing'],
    ['wc-completed', 'wc-processing', 'completed', 'processing'],
    ['completed', 'processing'],
    ['wc-completed'],
    ['completed']
];

foreach ($statuses_to_try as $status_list) {
    $status_str = "'" . implode("', '", $status_list) . "'";
    $count = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*)
        FROM {$wpdb->prefix}wc_orders o
        JOIN {$wpdb->prefix}wc_orders_meta oi ON o.id = oi.order_id AND oi.meta_key = '_tribe_wooticket_for_event'
        WHERE oi.meta_value = %s AND o.status IN ($status_str)
    ", $event_id));
    
    echo "- Statuses [$status_str]: $count orders\n";
}

// Step 3: Get actual order statuses
echo "\n3️⃣ Actual Order Statuses:\n";
$actual_statuses = $wpdb->get_results($wpdb->prepare("
    SELECT o.status, COUNT(*) as count
    FROM {$wpdb->prefix}wc_orders o
    JOIN {$wpdb->prefix}wc_orders_meta oi ON o.id = oi.order_id AND oi.meta_key = '_tribe_wooticket_for_event'
    WHERE oi.meta_value = %s
    GROUP BY o.status
", $event_id));

foreach ($actual_statuses as $status) {
    echo "- '{$status->status}': {$status->count} orders\n";
}

// Step 4: Test the corrected sales calculation using HPOS
echo "\n4️⃣ Testing Corrected Sales Calculation (HPOS):\n";
$sales_data = $wpdb->get_results($wpdb->prepare("
    SELECT 
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event_id));

$paid_tickets = 0;
$free_tickets = 0;
$table_entries = 0;
$table_threshold = 100;

echo "Found " . count($sales_data) . " sales records\n";
foreach ($sales_data as $sale) {
    $price = (float) $sale->ticket_price;
    $qty = (int) $sale->quantity;
    $line_total = (float) $sale->line_total;
    
    if ($price >= $table_threshold) {
        $table_entries += $qty * 6;
        echo "Table: £$price x$qty = £$line_total → {$qty} tables = " . ($qty * 6) . " entries\n";
    } elseif ($line_total > 0) {
        $paid_tickets += $qty;
        echo "Paid: £$price x$qty = £$line_total\n";
    } else {
        $free_tickets += $qty;
        echo "Free: £$price x$qty = £$line_total\n";
    }
}

$minimum_forecast = $paid_tickets + $table_entries;
echo "\nSummary:\n";
echo "Paid tickets: $paid_tickets\n";
echo "Free tickets: $free_tickets\n";
echo "Table entries: $table_entries\n";
echo "Minimum forecast: $minimum_forecast\n";

// Step 5: Test forecasting engine
echo "\n5️⃣ Testing Forecasting Engine:\n";
try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    $forecast = $forecasting_engine->generate_forecast($event_id);
    
    if ($forecast && isset($forecast['forecast'])) {
        $forecast_tickets = $forecast['forecast']['ticket_sales_p50'] ?? 0;
        echo "Generated forecast:\n";
        echo "- Ticket sales p50: $forecast_tickets\n";
        echo "- Attendance p50: " . ($forecast['forecast']['attendance_p50'] ?? 'N/A') . "\n";
        echo "- Confidence: " . ($forecast['forecast']['confidence'] ?? 'N/A') . "\n";
        echo "- Cohort count: " . ($forecast['forecast']['cohort_count'] ?? 'N/A') . "\n";
        
        echo "\nRule Check:\n";
        $forecast_attendance = $forecast['forecast']['attendance_p50'] ?? 0;
        echo "Ticket sales p50: $forecast_tickets (can be lower than current sales)\n";
        echo "Attendance p50: $forecast_attendance (should be >= paid + tables)\n";
        echo "Paid tickets + table entries: $minimum_forecast\n";
        
        if ($forecast_attendance >= $minimum_forecast) {
            echo "✅ SUCCESS: Attendance forecast ($forecast_attendance) >= paid tickets + table entries ($minimum_forecast)\n";
        } else {
            echo "❌ FAILED: Attendance forecast ($forecast_attendance) < paid tickets + table entries ($minimum_forecast)\n";
        }
    } else {
        echo "❌ Failed to generate forecast\n";
    }
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Comprehensive test complete.\n";
?>
