<?php
/**
 * Plugin Name: Coco Ops Forecast & Staffing Recommender
 * Description: Operations intelligence plugin for forecasting attendance and recommending staffing levels based on historical data and ticket sales.
 * Version: 1.0.0
 * Author: Cococure
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('COCO_OPS_VERSION', '1.0.0');
define('COCO_OPS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('COCO_OPS_PLUGIN_URL', plugin_dir_url(__FILE__));

class Coco_Ops_Forecast {
    private static $instance = null;
    const SNAPSHOT_CRON_HOOK = 'coco_ops_snapshot_cron';
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    private function load_dependencies() {
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-database.php';
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-rest-api.php';
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-rules-engine.php';
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-cohort-matcher.php';
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-forecasting-engine.php';
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-snapshot-cron.php';
        require_once COCO_OPS_PLUGIN_DIR . 'includes/class-post-event-report.php';
    }
    
    private function init_hooks() {
        // Register capabilities
        add_action('admin_init', [$this, 'register_capabilities']);
        
        // Admin menu
        add_action('admin_menu', [$this, 'register_admin_menu']);
        
        // Enqueue admin assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        
        // Initialize REST API
        add_action('rest_api_init', [new Coco_Ops_REST_API(), 'register_routes']);
        
        // Register cron schedules
        add_filter('cron_schedules', [$this, 'add_cron_schedules']);
        
        // Register snapshot cron
        add_action(self::SNAPSHOT_CRON_HOOK, [new Coco_Ops_Snapshot_Cron(), 'run_snapshots']);
        add_action('coco_ops_post_event_snapshots', [new Coco_Ops_Snapshot_Cron(), 'generate_post_event_snapshots']);
        
        // Register AJAX handlers
        add_action('wp_ajax_coco_ops_generate_historical_snapshots', 'coco_ops_generate_historical_snapshots');
    }
    
    public function register_capabilities() {
        $admin_role = get_role('administrator');
        if ($admin_role) {
            $admin_role->add_cap('coco_ops_view');
            $admin_role->add_cap('coco_ops_admin');
        }
    }
    
    public function add_cron_schedules($schedules) {
        if (!isset($schedules['twice_daily'])) {
            $schedules['twice_daily'] = [
                'interval' => 12 * HOUR_IN_SECONDS,
                'display' => __('Twice Daily', 'coco-ops-forecast')
            ];
        }
        return $schedules;
    }
    
    public function register_admin_menu() {
        // Main menu
        add_menu_page(
            __('Ops Forecast', 'coco-ops-forecast'),
            __('Ops Forecast', 'coco-ops-forecast'),
            'coco_ops_view',
            'coco-ops-forecast',
            [$this, 'render_events_page'],
            'dashicons-chart-line',
            30
        );
        
        // Events page (default)
        add_submenu_page(
            'coco-ops-forecast',
            __('Events', 'coco-ops-forecast'),
            __('Events', 'coco-ops-forecast'),
            'coco_ops_view',
            'coco-ops-forecast',
            [$this, 'render_events_page']
        );
        
        // Forecast Dashboard
        add_submenu_page(
            'coco-ops-forecast',
            __('Forecast Dashboard', 'coco-ops-forecast'),
            __('Forecast Dashboard', 'coco-ops-forecast'),
            'coco_ops_view',
            'coco-ops-forecast-dashboard',
            [$this, 'render_forecast_dashboard']
        );
        
        // Rules
        add_submenu_page(
            'coco-ops-forecast',
            __('Rules', 'coco-ops-forecast'),
            __('Rules', 'coco-ops-forecast'),
            'coco_ops_admin',
            'coco-ops-rules',
            [$this, 'render_rules_page']
        );
        
        // Settings
        add_submenu_page(
            'coco-ops-forecast',
            __('Settings', 'coco-ops-forecast'),
            __('Settings', 'coco-ops-forecast'),
            'coco_ops_admin',
            'coco-ops-settings',
            [$this, 'render_settings_page']
        );
        
        // Reports
        add_submenu_page(
            'coco-ops-forecast',
            __('Reports', 'coco-ops-forecast'),
            __('Reports', 'coco-ops-forecast'),
            'coco_ops_view',
            'coco-ops-reports',
            [$this, 'admin_page_reports']
        );
        
        // Debug Reports
        add_submenu_page(
            'coco-ops-forecast',
            __('Debug Reports', 'coco-ops-forecast'),
            __('Debug Reports', 'coco-ops-forecast'),
            'coco_ops_admin',
            'coco-ops-debug-reports',
            [$this, 'admin_page_debug_reports']
        );
        
        // About
        add_submenu_page(
            'coco-ops-forecast',
            __('About', 'coco-ops-forecast'),
            __('About', 'coco-ops-forecast'),
            'coco_ops_view',
            'coco-ops-about',
            [$this, 'render_about_page']
        );
        
        // Debug (only for admins)
        add_submenu_page(
            'coco-ops-forecast',
            __('Debug', 'coco-ops-forecast'),
            __('Debug', 'coco-ops-forecast'),
            'coco_ops_admin',
            'coco-ops-debug',
            [$this, 'render_debug_page']
        );
        
        // Venue Diagnostic (only for admins)
        add_submenu_page(
            'coco-ops-forecast',
            __('Venue Diagnostic', 'coco-ops-forecast'),
            __('Venue Diagnostic', 'coco-ops-forecast'),
            'coco_ops_admin',
            'coco-ops-venue-diagnostic',
            [$this, 'render_venue_diagnostic_page']
        );
    }
    
    public function enqueue_admin_assets($hook) {
        // Only load on our plugin pages
        if (strpos($hook, 'coco-ops') === false) {
            return;
        }
        
        // Tailwind CSS
        wp_enqueue_style(
            'coco-ops-tailwind',
            'https://cdn.jsdelivr.net/npm/tailwindcss@3.4.0/dist/tailwind.min.css',
            [],
            COCO_OPS_VERSION
        );
        
        // Custom styles
        wp_enqueue_style(
            'coco-ops-admin',
            COCO_OPS_PLUGIN_URL . 'assets/css/admin-styles.css',
            ['coco-ops-tailwind'],
            COCO_OPS_VERSION
        );
        
        // Chart.js (load first)
        wp_enqueue_script(
            'coco-ops-chartjs',
            'https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js',
            [],
            '4.4.0',
            true
        );
        
        // Admin app (load second - registers Alpine components)
        wp_enqueue_script(
            'coco-ops-admin-app',
            COCO_OPS_PLUGIN_URL . 'assets/js/admin-app.js',
            ['coco-ops-chartjs'],
            COCO_OPS_VERSION . '-' . time() . '-v2', // Force cache refresh
            true
        );
        
        // Alpine.js (load LAST - after components are registered)
        wp_enqueue_script(
            'coco-ops-alpine',
            'https://cdn.jsdelivr.net/npm/alpinejs@3.13.0/dist/cdn.min.js',
            ['coco-ops-admin-app'],
            '3.13.0',
            true
        );
        
        // Localize script with REST API data
        wp_localize_script('coco-ops-admin-app', 'cocoOpsData', [
            'restUrl' => rest_url('coco-ops/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'adminUrl' => admin_url('admin.php?page=coco-ops-forecast'),
            'currentUser' => [
                'id' => get_current_user_id(),
                'canManageOptions' => current_user_can('manage_options'),
                'canEditPosts' => current_user_can('edit_posts')
            ]
        ]);
    }
    
    public function render_events_page() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/events-list.php';
    }
    
    public function render_forecast_dashboard() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/forecast-dashboard.php';
    }
    
    public function render_rules_page() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/rules.php';
    }
    
    public function render_settings_page() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/settings.php';
    }
    
    public function render_about_page() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/about.php';
    }
    
    public function render_debug_page() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/debug.php';
    }
    
    public function render_venue_diagnostic_page() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/venue-diagnostic.php';
    }
    
    public function admin_page_reports() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/reports-dashboard.php';
    }
    
    public function admin_page_debug_reports() {
        include COCO_OPS_PLUGIN_DIR . 'templates/admin-pages/debug-reports.php';
    }
}

// Activation hook
register_activation_hook(__FILE__, function() {
    // Create database tables
    Coco_Ops_Database::create_tables();
    
    // Seed default ruleset
    Coco_Ops_Database::seed_default_ruleset();
    
    // Schedule cron jobs
    if (!wp_next_scheduled(Coco_Ops_Forecast::SNAPSHOT_CRON_HOOK)) {
        wp_schedule_event(time(), 'twice_daily', Coco_Ops_Forecast::SNAPSHOT_CRON_HOOK);
    }
    
    if (!wp_next_scheduled('coco_ops_post_event_snapshots')) {
        wp_schedule_event(time(), 'daily', 'coco_ops_post_event_snapshots');
    }
    
    // Add capabilities to administrator role
    $admin_role = get_role('administrator');
    if ($admin_role) {
        $admin_role->add_cap('coco_ops_view');
        $admin_role->add_cap('coco_ops_admin');
    }
    
    // Add view capability to editor role
    $editor_role = get_role('editor');
    if ($editor_role) {
        $editor_role->add_cap('coco_ops_view');
    }
    
    // Flush rewrite rules
    flush_rewrite_rules();
});

// Deactivation hook
register_deactivation_hook(__FILE__, function() {
    // Clear scheduled cron
    wp_clear_scheduled_hook(Coco_Ops_Forecast::SNAPSHOT_CRON_HOOK);
    
    // Remove capabilities
    $admin_role = get_role('administrator');
    if ($admin_role) {
        $admin_role->remove_cap('coco_ops_view');
        $admin_role->remove_cap('coco_ops_admin');
    }
    
    $editor_role = get_role('editor');
    if ($editor_role) {
        $editor_role->remove_cap('coco_ops_view');
    }
    
    // Flush rewrite rules
    flush_rewrite_rules();
});

// AJAX handler for generating historical snapshots
function coco_ops_generate_historical_snapshots() {

    // Log to file directly
    $log_file = ABSPATH . 'wp-content/coco-ops-debug.log';
    file_put_contents($log_file, "[" . current_time('mysql') . "] AJAX handler called\n", FILE_APPEND);
    file_put_contents($log_file, "[" . current_time('mysql') . "] POST data: " . print_r($_POST, true) . "\n", FILE_APPEND);
    
    error_log('CocoOpsHistorical: AJAX handler called at ' . current_time('mysql'));
    error_log('CocoOpsHistorical: POST data: ' . print_r($_POST, true));
    
    // Simple test to verify AJAX is working
    if (isset($_POST['test_only']) && $_POST['test_only'] == '1') {
        error_log('CocoOpsHistorical: Test mode - returning success');
        wp_send_json_success([
            'processed' => 0,
            'snapshots_created' => 0,
            'time_taken' => 0.1,
            'message' => 'AJAX handler is working!'
        ]);
        return;
    }
    
    error_log('CocoOpsHistorical: Starting real processing...');
    file_put_contents($log_file, "[" . current_time('mysql') . "] Starting real processing...\n", FILE_APPEND);
    
    // Verify nonce
    if (!wp_verify_nonce($_POST['nonce'], 'coco_ops_historical_snapshots')) {
        error_log('CocoOpsHistorical: Nonce verification failed');
        wp_send_json_error(['message' => 'Security check failed']);
        return;
    }
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Unauthorized']);
        return;
    }
    
    // Quick test - just return success without processing
    // error_log('CocoOpsHistorical: Quick test - returning success');
    // wp_send_json_success([
    //     'processed' => 0,
    //     'snapshots_created' => 0,
    //     'time_taken' => 0.1,
    //     'message' => 'Quick test successful - AJAX handler is working'
    // ]);
    // return;
    
    global $wpdb;
    
    // Set execution time limit
    set_time_limit(60); // 60 seconds max
    
    $batch_size = intval($_POST['batch_size'] ?? 5);
    $start_time = microtime(true);
    
    try {
        error_log('CocoOpsHistorical: Starting processing with batch size: ' . $batch_size);
        file_put_contents($log_file, "[" . current_time('mysql') . "] Starting processing with batch size: {$batch_size}\n", FILE_APPEND);
        
        // Get the OLDEST historical events that have products linked
        $events = $wpdb->get_results($wpdb->prepare("
            SELECT DISTINCT p.ID, p.post_title, pm_start.meta_value as start_date
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
            INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
            WHERE p.post_type = 'tribe_events' 
            AND p.post_status = 'publish'
            AND pm_start.meta_value < NOW()
            ORDER BY pm_start.meta_value ASC
            LIMIT %d
        ", $batch_size));
        
        error_log('CocoOpsHistorical: Found ' . count($events) . ' events to process');
        file_put_contents($log_file, "[" . current_time('mysql') . "] Found " . count($events) . " events to process\n", FILE_APPEND);
        
        if (empty($events)) {
            error_log('CocoOpsHistorical: No events found to process');
            file_put_contents($log_file, "[" . current_time('mysql') . "] No events found to process\n", FILE_APPEND);
            wp_send_json_error(['message' => 'No events found to process']);
            return;
        }
        
        $processed = 0;
        $snapshots_created = 0;
        
        // Get settings
        $settings = get_option('coco_ops_settings', ['table_price_threshold' => 100.00]);
        $table_threshold = isset($settings['table_price_threshold']) ? (float) $settings['table_price_threshold'] : 100.00;
        
        // Check if snapshots table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}coco_event_snapshots'");
        if (!$table_exists) {
            error_log('CocoOpsHistorical: Snapshots table does not exist');
            wp_send_json_error(['message' => 'Snapshots table does not exist. Please activate the plugin first.']);
            return;
        }
        
        // Check if staffing table exists
        $staffing_table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}coco_staffing_recs'");
        if (!$staffing_table_exists) {
            error_log('CocoOpsHistorical: Staffing table does not exist');
            wp_send_json_error(['message' => 'Staffing table does not exist. Please activate the plugin first.']);
            return;
        }
        
        error_log('CocoOpsHistorical: Snapshots table exists, proceeding with processing');
        
        foreach ($events as $event) {
            error_log('CocoOpsHistorical: Processing event: ' . $event->post_title . ' (ID: ' . $event->ID . ')');
            file_put_contents($log_file, "[" . current_time('mysql') . "] Processing event: {$event->post_title} (ID: {$event->ID})\n", FILE_APPEND);
            
            // Check if event has products linked
            $products_linked = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->postmeta}
                WHERE meta_key = '_tribe_wooticket_for_event'
                AND meta_value = %d
            ", $event->ID));
            
            error_log('CocoOpsHistorical: Event ' . $event->ID . ' has ' . $products_linked . ' products linked');
            
            if ($products_linked == 0) {
                error_log('CocoOpsHistorical: Skipping event ' . $event->ID . ' - no products linked');
                $processed++;
                continue;
            }
            
            // Calculate snapshot dates (D-14 to D-1) for historical events
            $event_start = new DateTime($event->start_date);
            $snapshot_dates = [];
            
            // Generate snapshots from D-14 to D-1 for historical events (before the event)
            for ($i = 14; $i >= 1; $i--) {
                $snapshot_date = clone $event_start;
                $snapshot_date->modify("-{$i} days");
                $snapshot_dates[] = $snapshot_date;
            }
            
            error_log('CocoOpsHistorical: Created ' . count($snapshot_dates) . ' snapshot dates for event ' . $event->ID . ' (event date: ' . $event->start_date . ')');
            file_put_contents($log_file, "[" . current_time('mysql') . "] Created " . count($snapshot_dates) . " snapshot dates for event {$event->ID}\n", FILE_APPEND);
            
            foreach ($snapshot_dates as $snapshot_date) {
                file_put_contents($log_file, "[" . current_time('mysql') . "] Processing snapshot date: " . $snapshot_date->format('Y-m-d H:i:s') . " for event {$event->ID}\n", FILE_APPEND);
                // Calculate days to event
                $days_to_event = $event_start->diff($snapshot_date)->days;
                if ($event_start < $snapshot_date) {
                    $days_to_event = -$days_to_event;
                }
                
                // Get sales data up to this snapshot date
                file_put_contents($log_file, "[" . current_time('mysql') . "] Getting sales data for event {$event->ID} at " . $snapshot_date->format('Y-m-d H:i:s') . "\n", FILE_APPEND);
                $sales_data = get_historical_sales_data($event->ID, $snapshot_date, $table_threshold);
                
                error_log('CocoOpsHistorical: Sales data for event ' . $event->ID . ' at ' . $snapshot_date->format('Y-m-d H:i:s') . ': ' . json_encode($sales_data));
                file_put_contents($log_file, "[" . current_time('mysql') . "] Sales data: " . json_encode($sales_data) . "\n", FILE_APPEND);
                
                // Check if we got any sales data
                if ($sales_data['tickets'] == 0 && $sales_data['tables'] == 0 && $sales_data['revenue'] == 0) {
                    error_log('CocoOpsHistorical: No sales data found for event ' . $event->ID . ' at ' . $snapshot_date->format('Y-m-d H:i:s'));
                }
                
                // Check if snapshot already exists
                $existing_snapshot = $wpdb->get_var($wpdb->prepare("
                    SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
                    WHERE event_id = %d AND snapshot_ts = %s
                ", $event->ID, $snapshot_date->format('Y-m-d H:i:s')));
                
                if ($existing_snapshot > 0) {
                    file_put_contents($log_file, "[" . current_time('mysql') . "] Snapshot already exists for event {$event->ID} at " . $snapshot_date->format('Y-m-d H:i:s') . "\n", FILE_APPEND);
                    continue;
                }
                
                // Insert snapshot
                $result = $wpdb->insert(
                    $wpdb->prefix . 'coco_event_snapshots',
                    [
                        'event_id' => $event->ID,
                        'snapshot_ts' => $snapshot_date->format('Y-m-d H:i:s'),
                        'days_to_event' => $days_to_event,
                        'tickets_sold' => $sales_data['tickets'],
                        'tables_sold' => $sales_data['tables'],
                        'revenue_to_date' => $sales_data['revenue']
                    ],
                    ['%d', '%s', '%d', '%d', '%d', '%f']
                );
                
                if ($result) {
                    $snapshots_created++;
                    error_log('CocoOpsHistorical: Successfully created snapshot for event ' . $event->ID . ' at ' . $snapshot_date->format('Y-m-d H:i:s') . ' with data: ' . json_encode($sales_data));
                    file_put_contents($log_file, "[" . current_time('mysql') . "] Successfully created snapshot for event {$event->ID}\n", FILE_APPEND);
                } else {
                    error_log('CocoOpsHistorical: Failed to create snapshot for event ' . $event->ID . ' at ' . $snapshot_date->format('Y-m-d H:i:s') . ': ' . $wpdb->last_error);
                    file_put_contents($log_file, "[" . current_time('mysql') . "] Failed to create snapshot for event {$event->ID}: " . $wpdb->last_error . "\n", FILE_APPEND);
                    error_log('CocoOpsHistorical: Insert data was: ' . json_encode([
                        'event_id' => $event->ID,
                        'snapshot_ts' => $snapshot_date->format('Y-m-d H:i:s'),
                        'days_to_event' => $days_to_event,
                        'tickets_sold' => $sales_data['tickets'],
                        'tables_sold' => $sales_data['tables'],
                        'revenue_snapshot' => $sales_data['revenue']
                    ]));
                }
            }
            
            $processed++;
        }
        
        $time_taken = round(microtime(true) - $start_time, 2);
        
        error_log('CocoOpsHistorical: Success - processed: ' . $processed . ', snapshots: ' . $snapshots_created . ', time: ' . $time_taken);
        
        wp_send_json_success([
            'processed' => $processed,
            'snapshots_created' => $snapshots_created,
            'time_taken' => $time_taken
        ]);
        
    } catch (Exception $e) {
        error_log('CocoOpsHistorical: Exception - ' . $e->getMessage());
        error_log('CocoOpsHistorical: Stack trace - ' . $e->getTraceAsString());
        
        wp_send_json_error([
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]);
    }
}

function get_historical_sales_data($event_id, $snapshot_date, $table_threshold) {
    global $wpdb;
    
    error_log("CocoOpsHistorical: Getting sales data for event {$event_id} up to {$snapshot_date->format('Y-m-d H:i:s')}");
    
    // Query orders created up to the snapshot date - optimized with LIMIT
    $orders = $wpdb->get_results($wpdb->prepare("
        SELECT 
            p_ticket.post_title as ticket_name,
            pm_price.meta_value as ticket_price,
            CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
            CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
        FROM {$wpdb->prefix}wc_orders wc_orders
        INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
            wc_orders.id = wc_order_items.order_id
            AND wc_order_items.order_item_type = 'line_item'
        )
        INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
            wc_order_items.order_item_id = oim_ticket.order_item_id
            AND oim_ticket.meta_key = '_product_id'
        )
        INNER JOIN {$wpdb->posts} p_ticket ON (
            oim_ticket.meta_value = p_ticket.ID
            AND p_ticket.post_type = 'product'
        )
        INNER JOIN {$wpdb->postmeta} pm_price ON (
            p_ticket.ID = pm_price.post_id
            AND pm_price.meta_key = '_price'
        )
        INNER JOIN {$wpdb->postmeta} pm_event ON (
            p_ticket.ID = pm_event.post_id
            AND pm_event.meta_key = '_tribe_wooticket_for_event'
        )
        LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
            wc_order_items.order_item_id = oim_qty.order_item_id
            AND oim_qty.meta_key = '_qty'
        )
        LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
            wc_order_items.order_item_id = oim_line_total.order_item_id
            AND oim_line_total.meta_key = '_line_total'
        )
        WHERE pm_event.meta_value = %d
        AND wc_orders.status IN ('wc-completed', 'wc-processing')
        AND wc_orders.status != 'wc-refunded'
        AND wc_orders.date_created_gmt <= %s
        AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
        AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
        LIMIT 1000
    ", $event_id, $snapshot_date->format('Y-m-d H:i:s')));
    
    error_log("CocoOpsHistorical: Query returned " . count($orders) . " orders");
    
    if (count($orders) > 0) {
        error_log("CocoOpsHistorical: Sample order data: " . json_encode($orders[0]));
    }
    
    $tickets = 0;
    $tables = 0;
    $revenue = 0;
    
    // Categorize and sum up
    foreach ($orders as $order) {
        $price = (float) $order->ticket_price;
        $qty = (int) $order->quantity;
        $line_total = (float) $order->line_total;
        
        error_log("CocoOpsHistorical: Processing order - Price: {$price}, Qty: {$qty}, Line Total: {$line_total}, Threshold: {$table_threshold}");
        
        // Categorize based on threshold
        if ($price >= $table_threshold) {
            $tables += $qty;
            error_log("CocoOpsHistorical: Added {$qty} tables (price {$price} >= threshold {$table_threshold})");
        } else {
            $tickets += $qty;
            error_log("CocoOpsHistorical: Added {$qty} tickets (price {$price} < threshold {$table_threshold})");
        }
        
        $revenue += $line_total;
    }
    
    error_log("CocoOpsHistorical: Final totals - Tickets: {$tickets}, Tables: {$tables}, Revenue: {$revenue}");
    
    return [
        'tickets' => $tickets,
        'tables' => $tables,
        'revenue' => $revenue
    ];
}

// Initialize plugin
Coco_Ops_Forecast::get_instance();

