<?php
// Clean, readable forecast analysis for event 240721
require_once('../../../wp-config.php');

$event_id = 240721;

echo "🎯 FORECAST ANALYSIS: TWNTY7 - Grown & Sexy Saturdays (Event $event_id)\n";
echo "📅 Analysis Date: " . date('Y-m-d H:i:s') . "\n";
echo str_repeat("=", 80) . "\n\n";

global $wpdb;

// 1. EVENT OVERVIEW
echo "📋 EVENT OVERVIEW\n";
echo str_repeat("-", 50) . "\n";
$event = get_post($event_id);
$event_start = get_post_meta($event_id, '_EventStartDate', true);
$venue_id = get_post_meta($event_id, '_EventVenueID', true);
$venue = get_post($venue_id);

echo "Event: {$event->post_title}\n";
echo "Date: $event_start\n";
echo "Venue: " . ($venue ? $venue->post_title : 'Unknown') . "\n";

// Calculate event status
$start_dt = new DateTime($event_start);
$now_dt = new DateTime();
$days_diff = $start_dt->diff($now_dt)->days;
if ($start_dt < $now_dt) {
    $days_diff = -$days_diff;
}
echo "Status: " . ($days_diff < 0 ? "COMPLETED (D$days_diff)" : "UPCOMING (D$days_diff)") . "\n\n";

// 2. CURRENT SALES SUMMARY
echo "💰 CURRENT SALES SUMMARY\n";
echo str_repeat("-", 50) . "\n";

$live_sales = $wpdb->get_results($wpdb->prepare("
    SELECT 
        pm_price.meta_value as ticket_price,
        CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) as quantity,
        CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) as line_total
    FROM {$wpdb->prefix}wc_orders wc_orders
    INNER JOIN {$wpdb->prefix}woocommerce_order_items wc_order_items ON (
        wc_orders.id = wc_order_items.order_id
        AND wc_order_items.order_item_type = 'line_item'
    )
    INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_ticket ON (
        wc_order_items.order_item_id = oim_ticket.order_item_id
        AND oim_ticket.meta_key = '_product_id'
    )
    INNER JOIN {$wpdb->posts} p_ticket ON (
        oim_ticket.meta_value = p_ticket.ID
        AND p_ticket.post_type = 'product'
    )
    INNER JOIN {$wpdb->postmeta} pm_price ON (
        p_ticket.ID = pm_price.post_id
        AND pm_price.meta_key = '_price'
    )
    INNER JOIN {$wpdb->postmeta} pm_event ON (
        p_ticket.ID = pm_event.post_id
        AND pm_event.meta_key = '_tribe_wooticket_for_event'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_qty ON (
        wc_order_items.order_item_id = oim_qty.order_item_id
        AND oim_qty.meta_key = '_qty'
    )
    LEFT JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim_line_total ON (
        wc_order_items.order_item_id = oim_line_total.order_item_id
        AND oim_line_total.meta_key = '_line_total'
    )
    WHERE pm_event.meta_value = %d
    AND wc_orders.status IN ('wc-completed', 'wc-processing')
    AND wc_orders.status != 'wc-refunded'
    AND CAST(COALESCE(oim_qty.meta_value, 1) AS SIGNED) > 0
    AND CAST(COALESCE(oim_line_total.meta_value, 0) AS DECIMAL(10,2)) >= 0
", $event_id));

$total_tickets = 0;
$total_tables = 0;
$total_revenue = 0;
$paid_tickets = 0;
$free_tickets = 0;
$table_entries = 0;

foreach ($live_sales as $sale) {
    $price = (float) $sale->ticket_price;
    $qty = (int) $sale->quantity;
    $total = (float) $sale->line_total;
    
    $total_tickets += $qty;
    $total_revenue += $total;
    
    if ($price >= 100) {
        $total_tables += $qty;
        $table_entries += $qty * 6;
    } elseif ($price > 0) {
        $paid_tickets += $qty;
    } else {
        $free_tickets += $qty;
    }
}

echo "📊 Sales Breakdown:\n";
echo "  • Total Tickets Sold: $total_tickets\n";
echo "  • Paid Tickets: $paid_tickets\n";
echo "  • Free Tickets: $free_tickets\n";
echo "  • Tables Sold: $total_tables (30 entries)\n";
echo "  • Total Revenue: £" . number_format($total_revenue, 2) . "\n";
echo "  • Paid + Table Entries: " . ($paid_tickets + $table_entries) . "\n\n";

// 3. COHORT ANALYSIS
echo "👥 COHORT ANALYSIS\n";
echo str_repeat("-", 50) . "\n";

$features = $wpdb->get_row($wpdb->prepare("
    SELECT * FROM {$wpdb->prefix}coco_event_features 
    WHERE event_id = %d
", $event_id));

// Use same time bucket logic as cohort matcher
$time_buckets = [];
switch ($features->start_bucket) {
    case 'late_night':
        $time_buckets = ['late_night', 'night'];
        break;
    case 'night':
        $time_buckets = ['night', 'late_night'];
        break;
    case 'evening':
        $time_buckets = ['evening', 'night'];
        break;
    case 'afternoon':
        $time_buckets = ['afternoon', 'evening'];
        break;
    case 'morning':
        $time_buckets = ['morning', 'afternoon'];
        break;
    default:
        $time_buckets = [$features->start_bucket];
}

echo "🔍 Cohort Criteria:\n";
echo "  • Venue: TWNTY7 (ID: {$features->venue_id})\n";
echo "  • Day: {$features->weekday}\n";
echo "  • Time: {$features->start_bucket} (matches: " . implode(', ', $time_buckets) . ")\n\n";

$placeholders = implode(',', array_fill(0, count($time_buckets), '%s'));

$cohorts = $wpdb->get_results($wpdb->prepare("
    SELECT ef.event_id, ef.start_ts, p.post_title
    FROM {$wpdb->prefix}coco_event_features ef
    LEFT JOIN {$wpdb->posts} p ON ef.event_id = p.ID
    WHERE ef.venue_id = %d
    AND ef.weekday = %s
    AND ef.start_bucket IN ($placeholders)
    AND ef.start_ts < NOW()
    AND ef.event_id != %d
    ORDER BY ef.start_ts DESC
    LIMIT 10
", array_merge([$features->venue_id, $features->weekday], $time_buckets, [$event_id])));

echo "📈 Cohort Events Found: " . count($cohorts) . "\n\n";

if (count($cohorts) > 0) {
    echo "📅 Cohort Events:\n";
    foreach ($cohorts as $i => $cohort) {
        $is_past = strtotime($cohort->start_ts) < time();
        $status = $is_past ? "✅ PAST" : "❌ FUTURE";
        echo "  " . ($i + 1) . ". {$cohort->post_title} ({$cohort->start_ts}) - $status\n";
    }
    echo "\n";
    
    // Get final sales for cohorts
    $cohort_ids = array_column($cohorts, 'event_id');
    $placeholders = implode(',', array_fill(0, count($cohort_ids), '%d'));
    
    $final_sales_query = "
        SELECT event_id, tickets_sold, snapshot_ts, days_to_event
        FROM {$wpdb->prefix}coco_event_snapshots
        WHERE event_id IN ($placeholders)
        AND snapshot_ts = (
            SELECT MAX(snapshot_ts) 
            FROM {$wpdb->prefix}coco_event_snapshots s2 
            WHERE s2.event_id = {$wpdb->prefix}coco_event_snapshots.event_id
        )
    ";
    
    $final_sales_results = $wpdb->get_results($wpdb->prepare($final_sales_query, $cohort_ids));
    $cohort_final_sales = [];
    
    foreach ($final_sales_results as $result) {
        $cohort_final_sales[$result->event_id] = $result->tickets_sold;
    }
    
    echo "📊 Cohort Final Sales:\n";
    $final_sales_values = [];
    foreach ($cohorts as $cohort) {
        $final_tickets = $cohort_final_sales[$cohort->event_id] ?? 0;
        $final_sales_values[] = $final_tickets;
        echo "  • {$cohort->post_title}: $final_tickets tickets\n";
    }
    
    if (!empty($final_sales_values)) {
        sort($final_sales_values);
        $p25 = $final_sales_values[floor(count($final_sales_values) * 0.25)];
        $p50 = $final_sales_values[floor(count($final_sales_values) * 0.50)];
        $p75 = $final_sales_values[floor(count($final_sales_values) * 0.75)];
        
        echo "\n📈 Cohort Percentiles:\n";
        echo "  • P25: $p25 tickets\n";
        echo "  • P50: $p50 tickets\n";
        echo "  • P75: $p75 tickets\n";
    }
} else {
    echo "❌ No cohort events found!\n";
}
echo "\n";

// 4. FORECAST RESULTS
echo "🔮 FORECAST RESULTS\n";
echo str_repeat("-", 50) . "\n";

try {
    require_once('includes/class-forecasting-engine.php');
    $forecasting_engine = new Coco_Ops_Forecasting_Engine();
    $forecast = $forecasting_engine->generate_forecast($event_id);
    
    if ($forecast && isset($forecast['forecast'])) {
        echo "✅ Forecast Generated Successfully!\n\n";
        
        echo "🎫 TICKET SALES FORECAST:\n";
        echo "  • P25: " . ($forecast['forecast']['ticket_sales_p25'] ?? 'N/A') . " tickets\n";
        echo "  • P50: " . ($forecast['forecast']['ticket_sales_p50'] ?? 'N/A') . " tickets\n";
        echo "  • P75: " . ($forecast['forecast']['ticket_sales_p75'] ?? 'N/A') . " tickets\n\n";
        
        echo "👥 ATTENDANCE FORECAST:\n";
        echo "  • P25: " . ($forecast['forecast']['attendance_p25'] ?? 'N/A') . " attendees\n";
        echo "  • P50: " . ($forecast['forecast']['attendance_p50'] ?? 'N/A') . " attendees\n";
        echo "  • P75: " . ($forecast['forecast']['attendance_p75'] ?? 'N/A') . " attendees\n\n";
        
        echo "💰 REVENUE FORECAST:\n";
        echo "  • P50: £" . ($forecast['forecast']['revenue_p50'] ?? 'N/A') . "\n\n";
        
        echo "📊 FORECAST METADATA:\n";
        echo "  • Confidence: " . ($forecast['forecast']['confidence'] ?? 'N/A') . "\n";
        echo "  • Method: " . ($forecast['forecast']['method'] ?? 'N/A') . "\n";
        echo "  • Cohort Count: " . ($forecast['forecast']['cohort_count'] ?? 'N/A') . "\n";
        
        if (isset($forecast['current'])) {
            echo "\n📈 DATA USED FOR FORECAST:\n";
            echo "  • Current Tickets: " . ($forecast['current']['tickets_sold'] ?? 'N/A') . "\n";
            echo "  • Current Tables: " . ($forecast['current']['tables_sold'] ?? 'N/A') . "\n";
            echo "  • Current Revenue: £" . ($forecast['current']['revenue_to_date'] ?? 'N/A') . "\n";
            echo "  • Days to Event: " . ($forecast['current']['days_to_event'] ?? 'N/A') . "\n";
        }
        
    } else {
        echo "❌ Failed to generate forecast\n";
        if (isset($forecast['error'])) {
            echo "Error: " . $forecast['error'] . "\n";
        }
    }
    
} catch (Exception $e) {
    echo "❌ Exception: " . $e->getMessage() . "\n";
}

echo "\n" . str_repeat("=", 80) . "\n";
echo "✅ ANALYSIS COMPLETE\n";
?>
