<?php
/**
 * Check Historical Events for Cohort Matching
 */

// Include WordPress
require_once('../../../wp-config.php');

global $wpdb;

echo "🔍 Checking Historical Events for Cohort Matching\n";
echo "Time: " . current_time('mysql') . "\n\n";

// Current event features
$current_event_id = 256352;
$current_features = $wpdb->get_row($wpdb->prepare(
    "SELECT * FROM {$wpdb->prefix}coco_event_features WHERE event_id = %d",
    $current_event_id
));

echo "📅 Current event features:\n";
echo "  - Venue: {$current_features->venue_id}\n";
echo "  - Weekday: {$current_features->weekday}\n";
echo "  - Start bucket: {$current_features->start_bucket}\n";
echo "  - Start time: {$current_features->start_ts}\n\n";

// Find historical events with similar features
echo "🔍 Looking for historical events with similar features...\n\n";

// 1. Same venue, same weekday, same start bucket (strict match)
$strict_matches = $wpdb->get_results($wpdb->prepare("
    SELECT ef.event_id, ef.venue_id, ef.weekday, ef.start_bucket, ef.start_ts
    FROM {$wpdb->prefix}coco_event_features ef
    WHERE ef.event_id != %d
    AND ef.venue_id = %s
    AND ef.weekday = %s
    AND ef.start_bucket = %s
    AND ef.start_ts < NOW()
    ORDER BY ef.start_ts DESC
    LIMIT 10
", $current_event_id, $current_features->venue_id, $current_features->weekday, $current_features->start_bucket));

echo "📊 Strict matches (same venue + weekday + start bucket): " . count($strict_matches) . "\n";
foreach ($strict_matches as $event) {
    echo "  - Event {$event->event_id}: {$event->start_ts}\n";
}

// 2. Same venue, same weekday (medium match)
$medium_matches = $wpdb->get_results($wpdb->prepare("
    SELECT ef.event_id, ef.venue_id, ef.weekday, ef.start_bucket, ef.start_ts
    FROM {$wpdb->prefix}coco_event_features ef
    WHERE ef.event_id != %d
    AND ef.venue_id = %s
    AND ef.weekday = %s
    AND ef.start_ts < NOW()
    ORDER BY ef.start_ts DESC
    LIMIT 10
", $current_event_id, $current_features->venue_id, $current_features->weekday));

echo "\n📊 Medium matches (same venue + weekday): " . count($medium_matches) . "\n";
foreach ($medium_matches as $event) {
    echo "  - Event {$event->event_id}: {$event->start_bucket} {$event->start_ts}\n";
}

// 3. Same venue only (loose match)
$loose_matches = $wpdb->get_results($wpdb->prepare("
    SELECT ef.event_id, ef.venue_id, ef.weekday, ef.start_bucket, ef.start_ts
    FROM {$wpdb->prefix}coco_event_features ef
    WHERE ef.event_id != %d
    AND ef.venue_id = %s
    AND ef.start_ts < NOW()
    ORDER BY ef.start_ts DESC
    LIMIT 10
", $current_event_id, $current_features->venue_id));

echo "\n📊 Loose matches (same venue only): " . count($loose_matches) . "\n";
foreach ($loose_matches as $event) {
    echo "  - Event {$event->event_id}: {$event->weekday} {$event->start_bucket} {$event->start_ts}\n";
}

// 4. Check if these events have snapshots
if (!empty($strict_matches)) {
    $event_ids = array_column($strict_matches, 'event_id');
    $snapshots_count = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
        WHERE event_id IN (" . implode(',', array_fill(0, count($event_ids), '%d')) . ")
    ", ...$event_ids));
    
    echo "\n📸 Snapshots for strict matches: {$snapshots_count}\n";
}

// 5. Check total historical events
$total_historical = $wpdb->get_var("
    SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_features 
    WHERE start_ts < NOW()
");

echo "\n📈 Total historical events: {$total_historical}\n";

// 6. Check total snapshots
$total_snapshots = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots");
echo "📸 Total snapshots: {$total_snapshots}\n";

echo "\n🏁 Historical events check complete\n";
?>
