<?php
/**
 * Check what's actually saved in the database
 */

require_once('../../../wp-load.php');

if (!current_user_can('manage_options')) {
    die('Unauthorized');
}

echo '<h1>🔍 Check Database Values</h1>';
echo '<style>
    body { font-family: system-ui; padding: 20px; }
    .success { color: #059669; }
    .error { color: #DC2626; }
    .info { color: #3B82F6; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 5px; }
</style>';

global $wpdb;

try {
    // Get the event with planning data
    $table_planning = $wpdb->prefix . 'coco_event_planning';
    $planning_event = $wpdb->get_row("SELECT * FROM $table_planning WHERE est_attendance > 0 LIMIT 1");
    
    if (!$planning_event) {
        echo '<p class="error">❌ No planning data found</p>';
        exit;
    }
    
    $event_id = $planning_event->event_id;
    echo '<p>Checking event ' . $event_id . '</p>';
    
    // Check what's in the staffing table
    $table_staffing = $wpdb->prefix . 'coco_staffing_recs';
    $saved_staffing = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_staffing WHERE event_id = %d ORDER BY created_at DESC LIMIT 1",
        $event_id
    ));
    
    if ($saved_staffing) {
        echo '<h2>📊 Database Values (Raw)</h2>';
        echo '<pre>' . print_r($saved_staffing, true) . '</pre>';
        
        echo '<h2>📊 Database Values (Formatted)</h2>';
        echo '<ul>';
        echo '<li><strong>Security:</strong> "' . $saved_staffing->rec_security . '" (type: ' . gettype($saved_staffing->rec_security) . ')</li>';
        echo '<li><strong>Bartenders:</strong> "' . $saved_staffing->rec_bartenders . '" (type: ' . gettype($saved_staffing->rec_bartenders) . ')</li>';
        echo '<li><strong>Wait Staff:</strong> "' . $saved_staffing->rec_wait . '" (type: ' . gettype($saved_staffing->rec_wait) . ')</li>';
        echo '<li><strong>Bottle Girls:</strong> "' . $saved_staffing->rec_bottle_girls . '" (type: ' . gettype($saved_staffing->rec_bottle_girls) . ')</li>';
        echo '<li><strong>Managers:</strong> "' . $saved_staffing->rec_managers . '" (type: ' . gettype($saved_staffing->rec_managers) . ')</li>';
        echo '</ul>';
        
        // Check if values are null, empty, or 0
        echo '<h2>🔍 Value Analysis</h2>';
        $fields = ['rec_security', 'rec_bartenders', 'rec_wait', 'rec_bottle_girls', 'rec_managers'];
        foreach ($fields as $field) {
            $value = $saved_staffing->$field;
            echo '<p><strong>' . $field . ':</strong> ';
            if (is_null($value)) {
                echo 'NULL';
            } elseif ($value === '') {
                echo 'Empty string';
            } elseif ($value === 0) {
                echo 'Zero';
            } else {
                echo 'Value: ' . $value;
            }
            echo '</p>';
        }
        
    } else {
        echo '<p class="error">❌ No staffing data found in database</p>';
    }
    
    // Test the rules engine again to see what it returns
    echo '<h2>🧮 Rules Engine Test</h2>';
    $table_features = $wpdb->prefix . 'coco_event_features';
    $features = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_features WHERE event_id = %d",
        $event_id
    ));
    
    if ($features) {
        $rules_engine = new Coco_Ops_Rules_Engine();
        $result = $rules_engine->compute_staffing(
            $planning_event->est_attendance,
            [
                'weekday' => $features->weekday,
                'start_bucket' => $features->start_bucket
            ],
            $planning_event->est_tables ?? 0
        );
        
        echo '<h3>Rules Engine Output:</h3>';
        echo '<pre>' . print_r($result, true) . '</pre>';
        
        // Test the specific values that should be saved
        echo '<h3>Values to Save:</h3>';
        echo '<ul>';
        echo '<li><strong>Security:</strong> ' . ($result['staffing']['security'] ?? 'NOT SET') . '</li>';
        echo '<li><strong>Bartenders:</strong> ' . ($result['staffing']['bartenders'] ?? 'NOT SET') . '</li>';
        echo '<li><strong>Wait Staff:</strong> ' . ($result['staffing']['wait_staff'] ?? 'NOT SET') . '</li>';
        echo '<li><strong>Bottle Girls:</strong> ' . ($result['staffing']['bottle_girls'] ?? 'NOT SET') . '</li>';
        echo '<li><strong>Managers:</strong> ' . ($result['staffing']['managers'] ?? 'NOT SET') . '</li>';
        echo '</ul>';
    }
    
} catch (Exception $e) {
    echo '<p class="error">❌ Error: ' . $e->getMessage() . '</p>';
    echo '<pre>' . $e->getTraceAsString() . '</pre>';
}

echo '<hr>';
echo '<p>Database check complete.</p>';
?>
