<?php
/**
 * Background Historical Snapshots Processor
 * 
 * This script can be run via cron or manually to process historical snapshots
 * in batches without browser timeouts.
 */

// Include WordPress
require_once('../../../wp-config.php');

// Only allow admin users or CLI
if (!is_admin() && php_sapi_name() !== 'cli') {
    die('Access denied');
}

global $wpdb;

echo "🚀 Starting Background Historical Snapshots Processing\n";
echo "Time: " . current_time('mysql') . "\n\n";

// Configuration
$batch_size = 10; // Process 10 events at a time
$max_execution_time = 300; // 5 minutes max per run
$log_file = ABSPATH . 'wp-content/coco-ops-background.log';

// Set execution time limit
set_time_limit($max_execution_time);

// Log start
file_put_contents($log_file, "[" . current_time('mysql') . "] Background processor started\n", FILE_APPEND);

$start_time = microtime(true);
$total_processed = 0;
$total_snapshots = 0;

try {
    // Get total count of events to process
    $total_events = $wpdb->get_var("
        SELECT COUNT(DISTINCT p.ID)
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        AND pm_start.meta_value < NOW()
    ");
    
    echo "📊 Total events to process: {$total_events}\n";
    
    // Process in batches
    $offset = 0;
    $batch_number = 1;
    
    while ($offset < $total_events) {
        echo "\n🔄 Processing batch {$batch_number} (events {$offset} to " . ($offset + $batch_size) . ")\n";
        
        // Get batch of events
        $events = $wpdb->get_results($wpdb->prepare("
            SELECT DISTINCT p.ID, p.post_title, pm_start.meta_value as start_date
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
            INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
            WHERE p.post_type = 'tribe_events' 
            AND p.post_status = 'publish'
            AND pm_start.meta_value < NOW()
            ORDER BY pm_start.meta_value ASC
            LIMIT %d OFFSET %d
        ", $batch_size, $offset));
        
        if (empty($events)) {
            echo "✅ No more events to process\n";
            break;
        }
        
        $batch_processed = 0;
        $batch_snapshots = 0;
        
        foreach ($events as $event) {
            echo "  📅 Processing: {$event->post_title} (ID: {$event->ID})\n";
            
            // Check if event has products linked
            $products_linked = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->postmeta}
                WHERE meta_key = '_tribe_wooticket_for_event'
                AND meta_value = %d
            ", $event->ID));
            
            if ($products_linked == 0) {
                echo "    ⚠️  Skipping - no products linked\n";
                $batch_processed++;
                continue;
            }
            
            // Calculate snapshot dates (D-14 to D-1)
            $event_start = new DateTime($event->start_date);
            $snapshot_dates = [];
            
            for ($i = 14; $i >= 1; $i--) {
                $snapshot_date = clone $event_start;
                $snapshot_date->modify("-{$i} days");
                $snapshot_dates[] = $snapshot_date;
            }
            
            echo "    📸 Creating " . count($snapshot_dates) . " snapshots\n";
            
            foreach ($snapshot_dates as $snapshot_date) {
                // Check if snapshot already exists
                $existing = $wpdb->get_var($wpdb->prepare("
                    SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
                    WHERE event_id = %d AND snapshot_ts = %s
                ", $event->ID, $snapshot_date->format('Y-m-d H:i:s')));
                
                if ($existing > 0) {
                    continue; // Skip existing snapshots
                }
                
                // Get sales data
                $sales_data = get_historical_sales_data($event->ID, $snapshot_date, 100.00);
                
                // Calculate days to event
                $days_to_event = $event_start->diff($snapshot_date)->days;
                if ($event_start < $snapshot_date) {
                    $days_to_event = -$days_to_event;
                }
                
                // Insert snapshot
                $result = $wpdb->insert(
                    $wpdb->prefix . 'coco_event_snapshots',
                    [
                        'event_id' => $event->ID,
                        'snapshot_ts' => $snapshot_date->format('Y-m-d H:i:s'),
                        'days_to_event' => $days_to_event,
                        'tickets_sold' => $sales_data['tickets'],
                        'tables_sold' => $sales_data['tables'],
                        'revenue_to_date' => $sales_data['revenue']
                    ],
                    ['%d', '%s', '%d', '%d', '%d', '%f']
                );
                
                if ($result) {
                    $batch_snapshots++;
                }
            }
            
            $batch_processed++;
        }
        
        $total_processed += $batch_processed;
        $total_snapshots += $batch_snapshots;
        
        echo "  ✅ Batch {$batch_number} complete: {$batch_processed} events, {$batch_snapshots} snapshots\n";
        
        // Log progress
        file_put_contents($log_file, "[" . current_time('mysql') . "] Batch {$batch_number}: {$batch_processed} events, {$batch_snapshots} snapshots\n", FILE_APPEND);
        
        $offset += $batch_size;
        $batch_number++;
        
        // Check if we're running out of time
        if ((microtime(true) - $start_time) > ($max_execution_time - 30)) {
            echo "⏰ Approaching time limit, stopping gracefully\n";
            break;
        }
    }
    
    $time_taken = round(microtime(true) - $start_time, 2);
    
    echo "\n🎉 Background processing complete!\n";
    echo "📊 Total processed: {$total_processed} events\n";
    echo "📸 Total snapshots: {$total_snapshots}\n";
    echo "⏱️  Time taken: {$time_taken}s\n";
    
    // Log completion
    file_put_contents($log_file, "[" . current_time('mysql') . "] Background processing complete: {$total_processed} events, {$total_snapshots} snapshots, {$time_taken}s\n", FILE_APPEND);
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    file_put_contents($log_file, "[" . current_time('mysql') . "] Error: " . $e->getMessage() . "\n", FILE_APPEND);
}

echo "\n🏁 Background processor finished\n";
?>
