<?php
/**
 * Fully Automated Historical Snapshots Processor
 * 
 * This script runs continuously until all events are processed.
 * It can be started once and will complete everything automatically.
 */

// Include WordPress
require_once('../../../wp-config.php');

// Only allow CLI or admin users
if (php_sapi_name() !== 'cli' && !is_admin()) {
    die('Access denied - This script must be run from command line');
}

global $wpdb;

echo "🚀 Starting Fully Automated Historical Snapshots Processing\n";
echo "Time: " . current_time('mysql') . "\n";
echo "This will run until ALL events are processed...\n\n";

// Configuration
$batch_size = 5; // Smaller batches for stability
$delay_between_batches = 2; // 2 seconds between batches
$log_file = ABSPATH . 'wp-content/coco-ops-auto.log';

// Set execution time limit to 0 (unlimited)
set_time_limit(0);

// Log start
file_put_contents($log_file, "[" . current_time('mysql') . "] Auto processor started - will run until complete\n", FILE_APPEND);

$start_time = microtime(true);
$total_processed = 0;
$total_snapshots = 0;
$batch_number = 1;

try {
    // Get total count of events to process
    $total_events = $wpdb->get_var("
        SELECT COUNT(DISTINCT p.ID)
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        AND pm_start.meta_value < NOW()
    ");
    
    echo "📊 Total events to process: {$total_events}\n";
    echo "🔄 Processing in batches of {$batch_size} events\n";
    echo "⏱️  Delay between batches: {$delay_between_batches} seconds\n\n";
    
    $offset = 0;
    
    while ($offset < $total_events) {
        echo "🔄 Batch {$batch_number} - Processing events " . ($offset + 1) . " to " . min($offset + $batch_size, $total_events) . " of {$total_events}\n";
        
        // Get batch of events
        $events = $wpdb->get_results($wpdb->prepare("
            SELECT DISTINCT p.ID, p.post_title, pm_start.meta_value as start_date
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
            INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
            WHERE p.post_type = 'tribe_events' 
            AND p.post_status = 'publish'
            AND pm_start.meta_value < NOW()
            ORDER BY pm_start.meta_value ASC
            LIMIT %d OFFSET %d
        ", $batch_size, $offset));
        
        if (empty($events)) {
            echo "✅ No more events to process\n";
            break;
        }
        
        $batch_processed = 0;
        $batch_snapshots = 0;
        
        foreach ($events as $event) {
            echo "  📅 {$event->post_title} (ID: {$event->ID})\n";
            
            // Check if event has products linked
            $products_linked = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->postmeta}
                WHERE meta_key = '_tribe_wooticket_for_event'
                AND meta_value = %d
            ", $event->ID));
            
            if ($products_linked == 0) {
                echo "    ⚠️  Skipping - no products linked\n";
                $batch_processed++;
                continue;
            }
            
            // Calculate snapshot dates (D-14 to D-1)
            $event_start = new DateTime($event->start_date);
            $snapshot_dates = [];
            
            for ($i = 14; $i >= 1; $i--) {
                $snapshot_date = clone $event_start;
                $snapshot_date->modify("-{$i} days");
                $snapshot_dates[] = $snapshot_date;
            }
            
            $event_snapshots = 0;
            
            foreach ($snapshot_dates as $snapshot_date) {
                // Check if snapshot already exists
                $existing = $wpdb->get_var($wpdb->prepare("
                    SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
                    WHERE event_id = %d AND snapshot_ts = %s
                ", $event->ID, $snapshot_date->format('Y-m-d H:i:s')));
                
                if ($existing > 0) {
                    continue; // Skip existing snapshots
                }
                
                // Get sales data
                $sales_data = get_historical_sales_data($event->ID, $snapshot_date, 100.00);
                
                // Calculate days to event
                $days_to_event = $event_start->diff($snapshot_date)->days;
                if ($event_start < $snapshot_date) {
                    $days_to_event = -$days_to_event;
                }
                
                // Insert snapshot
                $result = $wpdb->insert(
                    $wpdb->prefix . 'coco_event_snapshots',
                    [
                        'event_id' => $event->ID,
                        'snapshot_ts' => $snapshot_date->format('Y-m-d H:i:s'),
                        'days_to_event' => $days_to_event,
                        'tickets_sold' => $sales_data['tickets'],
                        'tables_sold' => $sales_data['tables'],
                        'revenue_to_date' => $sales_data['revenue']
                    ],
                    ['%d', '%s', '%d', '%d', '%d', '%f']
                );
                
                if ($result) {
                    $event_snapshots++;
                    $batch_snapshots++;
                }
            }
            
            echo "    📸 Created {$event_snapshots} snapshots\n";
            $batch_processed++;
        }
        
        $total_processed += $batch_processed;
        $total_snapshots += $batch_snapshots;
        
        // Calculate progress
        $progress = round(($offset + $batch_size) / $total_events * 100, 1);
        $elapsed_time = round(microtime(true) - $start_time, 1);
        
        echo "  ✅ Batch complete: {$batch_processed} events, {$batch_snapshots} snapshots\n";
        echo "  📊 Progress: {$progress}% ({$total_processed}/{$total_events} events)\n";
        echo "  ⏱️  Elapsed time: {$elapsed_time}s\n";
        
        // Log progress
        file_put_contents($log_file, "[" . current_time('mysql') . "] Batch {$batch_number}: {$batch_processed} events, {$batch_snapshots} snapshots, Progress: {$progress}%\n", FILE_APPEND);
        
        $offset += $batch_size;
        $batch_number++;
        
        // Check if we're done
        if ($offset >= $total_events) {
            echo "\n🎉 ALL EVENTS PROCESSED!\n";
            break;
        }
        
        // Brief pause between batches
        if ($delay_between_batches > 0) {
            echo "  ⏳ Waiting {$delay_between_batches} seconds before next batch...\n";
            sleep($delay_between_batches);
        }
        
        echo "\n";
    }
    
    $time_taken = round(microtime(true) - $start_time, 2);
    
    echo "\n🎉 FULLY AUTOMATED PROCESSING COMPLETE!\n";
    echo "📊 Total processed: {$total_processed} events\n";
    echo "📸 Total snapshots: {$total_snapshots}\n";
    echo "⏱️  Total time: {$time_taken}s\n";
    echo "🏁 You can now close this terminal - everything is done!\n";
    
    // Log completion
    file_put_contents($log_file, "[" . current_time('mysql') . "] AUTO PROCESSING COMPLETE: {$total_processed} events, {$total_snapshots} snapshots, {$time_taken}s\n", FILE_APPEND);
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    file_put_contents($log_file, "[" . current_time('mysql') . "] Error: " . $e->getMessage() . "\n", FILE_APPEND);
}

echo "\n🏁 Auto processor finished\n";
?>
