// Coco Ops Forecast Admin App
// Defer Alpine.js initialization until components are registered
window.deferLoadingAlpine = function (callback) {
    document.addEventListener('DOMContentLoaded', callback);
};

(function() {
    'use strict';
    
    // API Service
    const API = {
        baseUrl: cocoOpsData.restUrl,
        nonce: cocoOpsData.nonce,
        
        async request(endpoint, options = {}) {
            const url = this.baseUrl + endpoint;
            const headers = {
                'Content-Type': 'application/json',
                'X-WP-Nonce': this.nonce
            };
            
            const config = {
                ...options,
                credentials: 'same-origin', // Ensure cookies are sent
                headers: { ...headers, ...options.headers }
            };
            
            console.log('🔍 API Request:', {
                url: url,
                nonce: this.nonce,
                credentials: config.credentials,
                currentUser: cocoOpsData.currentUser
            });
            
            try {
                const response = await fetch(url, config);
                
                // Check if response is ok before trying to parse JSON
                if (!response.ok) {
                    const errorText = await response.text();
                    console.error('❌ API Error Response:', {
                        status: response.status,
                        statusText: response.statusText,
                        body: errorText
                    });
                    
                    let errorData;
                    try {
                        errorData = JSON.parse(errorText);
                    } catch (e) {
                        errorData = { message: errorText };
                    }
                    
                    throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
                }
                
                const data = await response.json();
                
                console.log('📡 API Response:', {
                    url: url,
                    status: response.status,
                    ok: response.ok,
                    data: data
                });
                
                return data;
            } catch (error) {
                console.error('💥 API Error:', error);
                throw error;
            }
        },
        
        getEvents(filters = {}) {
            // Remove null/undefined values from filters
            const cleanFilters = Object.fromEntries(
                Object.entries(filters).filter(([_, value]) => value !== null && value !== undefined && value !== '')
            );
            const params = new URLSearchParams(cleanFilters);
            const queryString = params.toString();
            return this.request('events' + (queryString ? '?' + queryString : ''));
        },
        
        getEvent(id) {
            return this.request(`events/${id}`);
        },
        
        getEventForecast(id) {
            return this.request(`events/${id}/forecast`);
        },
        
        getEventSnapshots(id) {
            return this.request(`events/${id}/snapshots`);
        },
        
        getEventCohorts(id) {
            return this.request(`events/${id}/cohorts`);
        },
        
        computeStaffing(id, data) {
            return this.request(`events/${id}/compute-staffing`, {
                method: 'POST',
                body: JSON.stringify(data)
            });
        },
        
        savePlan(id, data) {
            return this.request(`events/${id}/save-plan`, {
                method: 'POST',
                body: JSON.stringify(data)
            });
        },
        
        saveActuals(id, data) {
            return this.request(`events/${id}/save-actuals`, {
                method: 'POST',
                body: JSON.stringify(data)
            });
        },
        
        getRulesets() {
            return this.request('rulesets');
        },
        
        getVenues() {
            return this.request('venues');
        },
        
        createRuleset(data) {
            return this.request('rulesets', {
                method: 'POST',
                body: JSON.stringify(data)
            });
        },
        
        activateRuleset(id) {
            return this.request(`rulesets/${id}/activate`, {
                method: 'POST'
            });
        },
        
        getSettings() {
            return this.request('settings');
        },
        
        saveSettings(data) {
            return this.request('settings', {
                method: 'POST',
                body: JSON.stringify(data)
            });
        },
        
        getVenues() {
            return this.request('venues');
        },
        
        getCurrentSales(eventId) {
            return this.request(`events/${eventId}/current-sales`);
        }
    };
    
    // Make API available globally
    window.CocoOpsAPI = API;
    
    // Alpine.js Components
    document.addEventListener('alpine:init', () => {
        
        // Events List Component
        Alpine.data('eventsList', () => ({
            events: [],
            venues: [],
            loading: true,
            filters: {
                venue: '',
                start_date: '',
                end_date: '',
                has_actuals: null
            },
            selectedEvent: null,
            showPlanModal: false,
            showActualsModal: false,
            
            async init() {
                // Set default filter to show current/future events
                this.filters.start_date = new Date().toISOString().split('T')[0];
                
                await Promise.all([
                    this.loadEvents(),
                    this.loadVenues()
                ]);
            },
            
            async loadVenues() {
                try {
                    const response = await API.getVenues();
                    
                    // Handle debug response
                    if (response.debug) {
                        console.log('Venue debug info:', response.debug);
                        this.venues = response.venues || [];
                    } else {
                        this.venues = response;
                    }
                } catch (error) {
                    console.error('Failed to load venues:', error);
                }
            },
            
            async loadEvents() {
                this.loading = true;
                try {
                    const response = await API.getEvents(this.filters);
                    
                    // Handle debug response
                    if (response.debug) {
                        console.log('Debug info:', response.debug);
                        if (response.debug.query_found_posts === 0) {
                            console.log('Total events in database:', response.debug.total_events);
                            if (response.debug.total_events && response.debug.total_events.publish > 0) {
                                alert('Events exist but query failed. Check the Debug page for more info.');
                            } else {
                                alert('No events found. Make sure:\n1. The Events Calendar plugin is active\n2. You have published events\n3. Events have start dates set');
                            }
                        }
                        this.events = response.events || [];
                    } else {
                        this.events = response;
                    }
                    // Enrich each event with live current sales and pacing signal
                    await this.enrichEventsWithLiveSalesAndPacing();
                } catch (error) {
                    console.error('Failed to load events:', error);
                    alert('Failed to load events. Please try again.');
                } finally {
                    this.loading = false;
                }
            },
            
            async enrichEventsWithLiveSalesAndPacing() {
                if (!Array.isArray(this.events) || this.events.length === 0) return;
                const limiter = async (tasks, concurrency = 4) => {
                    const results = [];
                    const executing = [];
                    for (const task of tasks) {
                        const p = task().then(r => { executing.splice(executing.indexOf(p), 1); return r; });
                        results.push(p);
                        executing.push(p);
                        if (executing.length >= concurrency) {
                            await Promise.race(executing);
                        }
                    }
                    return Promise.all(results);
                };
                const tasks = this.events.map((evt, idx) => async () => {
                    try {
                        const [forecast, current] = await Promise.all([
                            API.getEventForecast(evt.id),
                            API.getCurrentSales(evt.id)
                        ]);
                        this.$set(this.events[idx], 'current_live', current);
                        const pacingPct = forecast?.forecast?.pacing?.tickets_sold_vs_cohort || null;
                        this.$set(this.events[idx], 'pacing_pct', pacingPct);
                    } catch (e) {
                        console.warn('Enrich failed for event', evt.id, e);
                    }
                });
                await limiter(tasks, 4);
            },
            
            async applyFilters() {
                await this.loadEvents();
            },
            
            openPlanModal(event) {
                this.selectedEvent = event;
                this.showPlanModal = true;
            },
            
            openActualsModal(event) {
                this.selectedEvent = event;
                this.showActualsModal = true;
            },
            
            formatDate(dateString) {
                if (!dateString) return '-';
                const date = new Date(dateString);
                return date.toLocaleDateString('en-GB', {
                    day: 'numeric',
                    month: 'short',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
            },
            
            getStaffingSummary(staffing) {
                if (!staffing) return '-';
                return `${staffing.security}/${staffing.bottle_girls}/${staffing.wait_staff}/${staffing.bartenders}/${staffing.managers}`;
            }
        }));
        
        // Planning Modal Component
        Alpine.data('planningModal', (eventData) => ({
            mode: 'manual', // 'manual' or 'auto'
            est_attendance: eventData.planning?.est_attendance || '',
            est_tables: eventData.planning?.est_tables || '',
            notes: eventData.planning?.notes || '',
            staffing: null,
            forecast: null,
            loading: false,
            saving: false,
            
            async init() {
                if (this.mode === 'auto') {
                    await this.loadForecast();
                }
            },
            
            async switchMode(newMode) {
                this.mode = newMode;
                if (newMode === 'auto' && !this.forecast) {
                    await this.loadForecast();
                }
            },
            
            async loadForecast() {
                this.loading = true;
                try {
                    // Fetch both forecast and current sales data
                    const [forecast, currentSales] = await Promise.all([
                        API.getEventForecast(eventData.id),
                        API.getCurrentSales(eventData.id)
                    ]);
                    
                    // Merge current sales data with forecast
                    if (forecast && currentSales) {
                        forecast.current = currentSales;
                    }
                    
                    this.forecast = forecast;
                } catch (error) {
                    console.error('Failed to load forecast:', error);
                    alert('Failed to load forecast');
                } finally {
                    this.loading = false;
                }
            },
            
            async computeStaffing() {
                this.loading = true;
                try {
                    let attendance;
                    let tables = 0;
                    
                    if (this.mode === 'auto') {
                        if (!this.forecast || !this.forecast.forecast) {
                            alert('No forecast data available. Please try manual mode.');
                            return;
                        }
                        attendance = this.forecast.forecast.ticket_sales_p50;
                        // For auto mode, we don't have table estimates, so use 0
                        tables = 0;
                    } else {
                        attendance = parseInt(this.est_attendance);
                        tables = parseInt(this.est_tables) || 0;
                    }
                    
                    if (!attendance || attendance <= 0) {
                        alert('Please enter a valid attendance number');
                        return;
                    }
                    
                    const data = {
                        attendance: attendance,
                        tables: tables,
                        use_forecast: this.mode === 'auto'
                    };
                    
                    console.log('Computing staffing with data:', data);
                    this.staffing = await API.computeStaffing(eventData.id, data);
                } catch (error) {
                    console.error('Failed to compute staffing:', error);
                    alert('Failed to compute staffing: ' + error.message);
                } finally {
                    this.loading = false;
                }
            },
            
            async savePlan() {
                if (!this.staffing) {
                    alert('Please compute staffing first');
                    return;
                }
                
                this.saving = true;
                try {
                    let est_attendance;
                    let est_tables = 0;
                    
                    if (this.mode === 'auto') {
                        if (!this.forecast || !this.forecast.forecast) {
                            alert('No forecast data available. Please try manual mode.');
                            return;
                        }
                        est_attendance = this.forecast.forecast.ticket_sales_p50;
                        est_tables = 0; // Auto mode doesn't have table estimates
                    } else {
                        est_attendance = parseInt(this.est_attendance);
                        est_tables = parseInt(this.est_tables) || 0;
                    }
                    
                    const data = {
                        est_attendance: est_attendance,
                        est_tables: est_tables,
                        notes: this.notes,
                        use_forecast: this.mode === 'auto',
                        staffing: this.staffing
                    };
                    
                    console.log('Saving plan with data:', data);
                    await API.savePlan(eventData.id, data);
                    alert('Plan saved successfully!');
                    window.location.reload();
                } catch (error) {
                    console.error('Failed to save plan:', error);
                    alert('Failed to save plan: ' + error.message);
                } finally {
                    this.saving = false;
                }
            }
        }));
        
        // Actuals Modal Component
        Alpine.data('actualsModal', (eventData) => ({
            final_attendance: eventData.actuals?.final_attendance || '',
            final_revenue: eventData.actuals?.final_revenue || '',
            actual_bartenders: eventData.actuals?.actual_bartenders || '',
            actual_bottle_girls: eventData.actuals?.actual_bottle_girls || '',
            actual_wait: eventData.actuals?.actual_wait || '',
            actual_security: eventData.actuals?.actual_security || '',
            actual_managers: eventData.actuals?.actual_managers || '',
            notes: eventData.actuals?.notes || '',
            locked: eventData.actuals?.locked || false,
            saving: false,
            
            async saveActuals() {
                // Enhanced validation
                const validation = this.validateActuals();
                if (!validation.valid) {
                    alert(validation.message);
                    return;
                }
                
                this.saving = true;
                try {
                    const data = {
                        final_attendance: parseInt(this.final_attendance),
                        final_revenue: parseFloat(this.final_revenue) || null,
                        actual_bartenders: parseInt(this.actual_bartenders) || null,
                        actual_bottle_girls: parseInt(this.actual_bottle_girls) || null,
                        actual_wait: parseInt(this.actual_wait) || null,
                        actual_security: parseInt(this.actual_security) || null,
                        actual_managers: parseInt(this.actual_managers) || null,
                        notes: this.notes,
                        locked: this.locked
                    };
                    
                    await API.saveActuals(eventData.id, data);
                    alert('Actuals saved successfully!');
                    window.location.reload();
                } catch (error) {
                    console.error('Failed to save actuals:', error);
                    alert('Failed to save actuals');
                } finally {
                    this.saving = false;
                }
            },
            
            validateActuals() {
                // Required fields
                if (!this.final_attendance || this.final_attendance <= 0) {
                    return { valid: false, message: 'Final attendance is required and must be greater than 0' };
                }
                
                // Revenue validation
                if (this.final_revenue && this.final_revenue < 0) {
                    return { valid: false, message: 'Revenue cannot be negative' };
                }
                
                // Staff count validation
                const totalStaff = (parseInt(this.actual_bartenders) || 0) + 
                                 (parseInt(this.actual_bottle_girls) || 0) + 
                                 (parseInt(this.actual_wait) || 0) + 
                                 (parseInt(this.actual_security) || 0) + 
                                 (parseInt(this.actual_managers) || 0);
                
                if (totalStaff === 0) {
                    return { valid: false, message: 'At least one staff member must be entered' };
                }
                
                // Revenue per staff validation
                if (this.final_revenue && totalStaff > 0) {
                    const revenuePerStaff = this.final_revenue / totalStaff;
                    if (revenuePerStaff < 50) {
                        return { 
                            valid: false, 
                            message: `Revenue per staff is very low (£${revenuePerStaff.toFixed(2)}). Please verify the numbers.` 
                        };
                    }
                    if (revenuePerStaff > 1000) {
                        return { 
                            valid: false, 
                            message: `Revenue per staff is very high (£${revenuePerStaff.toFixed(2)}). Please verify the numbers.` 
                        };
                    }
                }
                
                // Attendance vs staff ratio validation
                const attendance = parseInt(this.final_attendance);
                const staffRatio = totalStaff / attendance;
                if (staffRatio > 0.1) { // More than 10% staff to attendance ratio
                    return { 
                        valid: false, 
                        message: `Staff ratio is very high (${(staffRatio * 100).toFixed(1)}%). Please verify the numbers.` 
                    };
                }
                
                // Forecast accuracy warning (not blocking)
                const estimated = eventData.planning?.est_attendance || 0;
                if (estimated > 0) {
                    const variance = Math.abs((attendance - estimated) / estimated * 100);
                    if (variance > 50) {
                        if (!confirm(`Forecast accuracy is low (${(100 - variance).toFixed(1)}% variance). Continue anyway?`)) {
                            return { valid: false, message: 'Save cancelled' };
                        }
                    }
                }
                
                return { valid: true };
            },
            
            getVariance() {
                const estimated = eventData.planning?.est_attendance || 0;
                const actual = parseInt(this.final_attendance) || 0;
                if (!estimated || !actual) return null;
                
                const variance = ((actual - estimated) / estimated * 100).toFixed(1);
                return {
                    value: variance,
                    positive: variance > 0
                };
            },
            
            calculateRevenuePerStaff() {
                const revenue = parseFloat(this.final_revenue) || 0;
                const totalStaff = (parseInt(this.actual_bartenders) || 0) + 
                                 (parseInt(this.actual_bottle_girls) || 0) + 
                                 (parseInt(this.actual_wait) || 0) + 
                                 (parseInt(this.actual_security) || 0) + 
                                 (parseInt(this.actual_managers) || 0);
                
                if (totalStaff === 0) return 'N/A';
                return '£' + (revenue / totalStaff).toFixed(2);
            },
            
            calculateStaffEfficiency() {
                const attendance = parseInt(this.final_attendance) || 0;
                const totalStaff = (parseInt(this.actual_bartenders) || 0) + 
                                 (parseInt(this.actual_bottle_girls) || 0) + 
                                 (parseInt(this.actual_wait) || 0) + 
                                 (parseInt(this.actual_security) || 0) + 
                                 (parseInt(this.actual_managers) || 0);
                
                if (attendance === 0) return 'N/A';
                return (totalStaff / attendance * 100).toFixed(1) + '%';
            },
            
            calculateForecastAccuracy() {
                const estimated = eventData.planning?.est_attendance || 0;
                const actual = parseInt(this.final_attendance) || 0;
                
                if (!estimated || !actual) return 'N/A';
                
                const accuracy = (100 - Math.abs((actual - estimated) / estimated * 100)).toFixed(1);
                return accuracy + '%';
            },
            
            calculateCapacityUtilization() {
                const attendance = parseInt(this.final_attendance) || 0;
                // This would need venue capacity data - for now return N/A
                return 'N/A';
            }
        }));
        
        // Forecast Dashboard Component
        Alpine.data('forecastDashboard', () => ({
            events: [],
            venues: [],
            selectedEventId: null,
            selectedEvent: null,
            forecast: null,
            snapshots: [],
            cohorts: [],
            loading: false,
            chart: null,
            
            async init() {
                await Promise.all([
                    this.loadEvents(),
                    this.loadVenues()
                ]);
            },
            
            async loadEvents() {
                try {
                    console.log('🔍 Forecast Dashboard: Loading events...');
                    // Load only current/future events for forecast dashboard
                    const today = new Date().toISOString().split('T')[0];
                    this.events = await API.getEvents({
                        start_date: today
                    });
                    console.log('📊 Forecast Dashboard: Loaded', this.events.length, 'events from', today);
                } catch (error) {
                    console.error('❌ Forecast Dashboard: Failed to load events:', error);
                }
            },
            
            async loadVenues() {
                try {
                    this.venues = await API.getVenues();
                } catch (error) {
                    console.error('Failed to load venues:', error);
                }
            },
            
            async selectEvent(eventId) {
                this.selectedEventId = eventId;
                this.loading = true;
                
                try {
                    console.log('🔍 Forecast Dashboard: Loading data for event', eventId);
                    
                    // Fetch current sales data AND forecast data
                    const [selectedEvent, currentSales, forecast, snapshots, cohorts] = await Promise.all([
                        API.getEvent(eventId),
                        API.getCurrentSales(eventId),
                        API.getEventForecast(eventId),
                        API.getEventSnapshots(eventId),
                        API.getEventCohorts(eventId)
                    ]);
                    
                    console.log('📊 Current sales data:', currentSales);
                    
                    this.selectedEvent = selectedEvent;
                    this.snapshots = snapshots;
                    this.cohorts = cohorts;
                    
                    // Merge current sales data with forecast data
                    // If forecast has no current data, use real-time sales data
                    if (!forecast.current || !forecast.current.tickets_sold) {
                        forecast.current = currentSales;
                    }
                    
                    this.forecast = forecast;
                    
                    console.log('📊 Forecast data loaded:', {
                        current: forecast.current,
                        forecast: forecast.forecast,
                        snapshots: snapshots.length,
                        cohorts: cohorts.length
                    });
                    
                    this.$nextTick(() => this.renderChart());
                } catch (error) {
                    console.error('Failed to load forecast data:', error);
                    alert('Failed to load forecast data');
                } finally {
                    this.loading = false;
                }
            },
            
            renderChart() {
                const canvas = document.getElementById('pacingChart');
                if (!canvas) return;
                
                if (this.chart) {
                    this.chart.destroy();
                }
                
                const ctx = canvas.getContext('2d');
                
                // Prepare data
                const labels = this.snapshots.map(s => {
                    return `D-${Math.round(s.days_to_event)}`;
                }).reverse();
                
                const ticketsData = this.snapshots.map(s => s.tickets_sold).reverse();
                
                this.chart = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: labels,
                        datasets: [{
                            label: 'Tickets Sold',
                            data: ticketsData,
                            borderColor: '#4F46E5',
                            backgroundColor: 'rgba(79, 70, 229, 0.1)',
                            tension: 0.4,
                            fill: true
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: true,
                                position: 'top'
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            },
            
            formatDate(dateString) {
                if (!dateString) return '-';
                const date = new Date(dateString);
                return date.toLocaleDateString('en-GB', {
                    day: 'numeric',
                    month: 'short',
                    year: 'numeric'
                });
            }
        }));
        
        // Rules Editor Component
        Alpine.data('rulesEditor', () => ({
            rulesets: [],
            activeRulesetId: null,
            newVersion: '',
            yamlContent: '',
            editing: false,
            saving: false,
            
            async init() {
                await this.loadRulesets();
            },
            
            async loadRulesets() {
                try {
                    this.rulesets = await API.getRulesets();
                    const active = this.rulesets.find(r => r.active === '1');
                    if (active) {
                        this.activeRulesetId = active.id;
                        this.yamlContent = active.yaml_content;
                    }
                } catch (error) {
                    console.error('Failed to load rulesets:', error);
                }
            },
            
            startEditing() {
                this.editing = true;
            },
            
            cancelEditing() {
                this.editing = false;
                const active = this.rulesets.find(r => r.active === '1');
                if (active) {
                    this.yamlContent = active.yaml_content;
                }
            },
            
            async saveRuleset() {
                if (!this.newVersion) {
                    alert('Please enter a version number');
                    return;
                }
                
                this.saving = true;
                try {
                    await API.createRuleset({
                        version: this.newVersion,
                        yaml_content: this.yamlContent
                    });
                    
                    alert('Ruleset saved! You can now activate it.');
                    await this.loadRulesets();
                    this.editing = false;
                    this.newVersion = '';
                } catch (error) {
                    console.error('Failed to save ruleset:', error);
                    alert('Failed to save ruleset: ' + error.message);
                } finally {
                    this.saving = false;
                }
            },
            
            async activateRuleset(id) {
                if (!confirm('Are you sure you want to activate this ruleset?')) {
                    return;
                }
                
                try {
                    await API.activateRuleset(id);
                    alert('Ruleset activated!');
                    await this.loadRulesets();
                } catch (error) {
                    console.error('Failed to activate ruleset:', error);
                    alert('Failed to activate ruleset');
                }
            }
        }));
        
        // Settings Component
        Alpine.data('settingsManager', () => ({
            settings: {
                snapshot_times: ['10:00', '18:00'],
                cohort_sensitivity: 'medium',
                enable_forecasting: true,
                min_cohort_size: 3
            },
            venues: [],
            saving: false,
            
            async init() {
                await Promise.all([
                    this.loadSettings(),
                    this.loadVenues()
                ]);
            },
            
            async loadSettings() {
                try {
                    this.settings = await API.getSettings();
                } catch (error) {
                    console.error('Failed to load settings:', error);
                }
            },
            
            async loadVenues() {
                try {
                    this.venues = await API.getVenues();
                } catch (error) {
                    console.error('Failed to load venues:', error);
                }
            },
            
            async saveSettings() {
                this.saving = true;
                try {
                    await API.saveSettings(this.settings);
                    alert('Settings saved successfully!');
                } catch (error) {
                    console.error('Failed to save settings:', error);
                    alert('Failed to save settings');
                } finally {
                    this.saving = false;
                }
            }
        }));
    });
    
    // Helper functions
    window.CocoOpsHelpers = {
        formatNumber(num) {
            return new Intl.NumberFormat('en-GB').format(num);
        },
        
        formatCurrency(amount) {
            return new Intl.NumberFormat('en-GB', {
                style: 'currency',
                currency: 'GBP'
            }).format(amount);
        },
        
        formatPercent(value) {
            return `${value > 0 ? '+' : ''}${value}%`;
        }
    };
    
})();

