<?php
/**
 * WordPress Admin Auto Processor
 * 
 * This creates an admin page that can run the auto processor
 * without command line access.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Add admin menu
add_action('admin_menu', 'coco_ops_auto_processor_menu');

function coco_ops_auto_processor_menu() {
    add_submenu_page(
        'coco-ops-forecast',
        'Auto Processor',
        'Auto Processor',
        'manage_options',
        'coco-ops-auto-processor',
        'coco_ops_auto_processor_page'
    );
}

function coco_ops_auto_processor_page() {
    ?>
    <div class="wrap">
        <h1>🚀 Auto Historical Snapshots Processor</h1>
        
        <div class="notice notice-info">
            <p><strong>⚠️ Important:</strong> This will process ALL historical events. It may take 2-4 hours to complete.</p>
        </div>
        
        <div id="auto-processor-status" style="display: none;">
            <h3>Processing Status</h3>
            <div id="progress-bar" style="width: 100%; background-color: #f0f0f0; border-radius: 5px; overflow: hidden;">
                <div id="progress-fill" style="width: 0%; height: 30px; background-color: #4CAF50; transition: width 0.3s;"></div>
            </div>
            <p id="progress-text">Starting...</p>
            <p id="progress-details"></p>
        </div>
        
        <div id="auto-processor-controls">
            <button id="start-auto-processor" class="button button-primary button-large">
                🚀 Start Auto Processing
            </button>
            <button id="stop-auto-processor" class="button button-secondary" style="display: none;">
                ⏹️ Stop Processing
            </button>
        </div>
        
        <div id="auto-processor-results" style="display: none;">
            <h3>Results</h3>
            <div id="results-content"></div>
        </div>
    </div>
    
    <script>
    let isProcessing = false;
    let currentBatch = 0;
    let totalBatches = 0;
    
    document.getElementById('start-auto-processor').addEventListener('click', function() {
        if (isProcessing) return;
        
        isProcessing = true;
        this.style.display = 'none';
        document.getElementById('stop-auto-processor').style.display = 'inline-block';
        document.getElementById('auto-processor-status').style.display = 'block';
        
        startAutoProcessing();
    });
    
    document.getElementById('stop-auto-processor').addEventListener('click', function() {
        isProcessing = false;
        this.style.display = 'none';
        document.getElementById('start-auto-processor').style.display = 'inline-block';
        document.getElementById('progress-text').textContent = 'Processing stopped by user';
    });
    
    async function startAutoProcessing() {
        try {
            // Get total events first
            const response = await fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'coco_ops_get_total_events',
                    nonce: '<?php echo wp_create_nonce('coco_ops_auto_processor'); ?>'
                })
            });
            
            const data = await response.json();
            if (data.success) {
                totalBatches = Math.ceil(data.data.total_events / 5); // 5 events per batch
                document.getElementById('progress-text').textContent = `Total events: ${data.data.total_events} (${totalBatches} batches)`;
            }
            
            // Start processing batches
            currentBatch = 0;
            await processBatches();
            
        } catch (error) {
            console.error('Error:', error);
            document.getElementById('progress-text').textContent = 'Error: ' + error.message;
        }
    }
    
    async function processBatches() {
        while (isProcessing && currentBatch < totalBatches) {
            try {
                const response = await fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'coco_ops_process_batch',
                        batch_number: currentBatch + 1,
                        batch_size: 5,
                        nonce: '<?php echo wp_create_nonce('coco_ops_auto_processor'); ?>'
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    currentBatch++;
                    const progress = Math.round((currentBatch / totalBatches) * 100);
                    
                    document.getElementById('progress-fill').style.width = progress + '%';
                    document.getElementById('progress-text').textContent = `Batch ${currentBatch}/${totalBatches} (${progress}%)`;
                    document.getElementById('progress-details').textContent = 
                        `Events: ${data.data.events_processed}, Snapshots: ${data.data.snapshots_created}`;
                    
                    if (currentBatch >= totalBatches) {
                        // Complete
                        isProcessing = false;
                        document.getElementById('stop-auto-processor').style.display = 'none';
                        document.getElementById('start-auto-processor').style.display = 'inline-block';
                        document.getElementById('auto-processor-results').style.display = 'block';
                        document.getElementById('results-content').innerHTML = 
                            `<p>✅ Processing complete! Processed ${data.data.total_events} events and created ${data.data.total_snapshots} snapshots.</p>`;
                    }
                } else {
                    throw new Error(data.data.message || 'Unknown error');
                }
                
                // Small delay between batches
                await new Promise(resolve => setTimeout(resolve, 1000));
                
            } catch (error) {
                console.error('Batch error:', error);
                document.getElementById('progress-text').textContent = 'Error in batch: ' + error.message;
                break;
            }
        }
    }
    </script>
    <?php
}

// AJAX handler to get total events
add_action('wp_ajax_coco_ops_get_total_events', 'coco_ops_get_total_events');

function coco_ops_get_total_events() {
    if (!wp_verify_nonce($_POST['nonce'], 'coco_ops_auto_processor')) {
        wp_send_json_error('Invalid nonce');
        return;
    }
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    global $wpdb;
    
    $total_events = $wpdb->get_var("
        SELECT COUNT(DISTINCT p.ID)
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        AND pm_start.meta_value < NOW()
    ");
    
    wp_send_json_success([
        'total_events' => intval($total_events)
    ]);
}

// AJAX handler to process a batch
add_action('wp_ajax_coco_ops_process_batch', 'coco_ops_process_batch');

function coco_ops_process_batch() {
    if (!wp_verify_nonce($_POST['nonce'], 'coco_ops_auto_processor')) {
        wp_send_json_error('Invalid nonce');
        return;
    }
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    $batch_number = intval($_POST['batch_number']);
    $batch_size = intval($_POST['batch_size']);
    $offset = ($batch_number - 1) * $batch_size;
    
    global $wpdb;
    
    // Get batch of events
    $events = $wpdb->get_results($wpdb->prepare("
        SELECT DISTINCT p.ID, p.post_title, pm_start.meta_value as start_date
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm_start ON (p.ID = pm_start.post_id AND pm_start.meta_key = '_EventStartDate')
        INNER JOIN {$wpdb->postmeta} pm_event ON (p.ID = pm_event.meta_value AND pm_event.meta_key = '_tribe_wooticket_for_event')
        WHERE p.post_type = 'tribe_events' 
        AND p.post_status = 'publish'
        AND pm_start.meta_value < NOW()
        ORDER BY pm_start.meta_value ASC
        LIMIT %d OFFSET %d
    ", $batch_size, $offset));
    
    $events_processed = 0;
    $snapshots_created = 0;
    
    foreach ($events as $event) {
        // Check if event has products linked
        $products_linked = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*)
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_tribe_wooticket_for_event'
            AND meta_value = %d
        ", $event->ID));
        
        if ($products_linked == 0) {
            $events_processed++;
            continue;
        }
        
        // Calculate snapshot dates (D-14 to D-1)
        $event_start = new DateTime($event->start_date);
        $snapshot_dates = [];
        
        for ($i = 14; $i >= 1; $i--) {
            $snapshot_date = clone $event_start;
            $snapshot_date->modify("-{$i} days");
            $snapshot_dates[] = $snapshot_date;
        }
        
        foreach ($snapshot_dates as $snapshot_date) {
            // Check if snapshot already exists
            $existing = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*) FROM {$wpdb->prefix}coco_event_snapshots 
                WHERE event_id = %d AND snapshot_ts = %s
            ", $event->ID, $snapshot_date->format('Y-m-d H:i:s')));
            
            if ($existing > 0) {
                continue; // Skip existing snapshots
            }
            
            // Get sales data
            $sales_data = get_historical_sales_data($event->ID, $snapshot_date, 100.00);
            
            // Calculate days to event
            $days_to_event = $event_start->diff($snapshot_date)->days;
            if ($event_start < $snapshot_date) {
                $days_to_event = -$days_to_event;
            }
            
            // Insert snapshot
            $result = $wpdb->insert(
                $wpdb->prefix . 'coco_event_snapshots',
                [
                    'event_id' => $event->ID,
                    'snapshot_ts' => $snapshot_date->format('Y-m-d H:i:s'),
                    'days_to_event' => $days_to_event,
                    'tickets_sold' => $sales_data['tickets'],
                    'tables_sold' => $sales_data['tables'],
                    'revenue_to_date' => $sales_data['revenue']
                ],
                ['%d', '%s', '%d', '%d', '%d', '%f']
            );
            
            if ($result) {
                $snapshots_created++;
            }
        }
        
        $events_processed++;
    }
    
    wp_send_json_success([
        'events_processed' => $events_processed,
        'snapshots_created' => $snapshots_created,
        'total_events' => $total_events ?? 0,
        'total_snapshots' => $snapshots_created
    ]);
}
?>
