# Permissions Fix - REST API 401 Error

## Problem
REST API endpoints were returning:
```json
{
  "code": "rest_forbidden",
  "message": "Sorry, you are not allowed to do that.",
  "data": { "status": 401 }
}
```

## Root Cause
The plugin's custom capabilities (`coco_ops_view` and `coco_ops_admin`) were not being added to user roles during plugin activation.

## Solution Applied

### 1. Updated Permission Callbacks ✅
Added fallback so administrators always have access even if capabilities aren't set:

```php
public function check_permission() {
    if (!is_user_logged_in()) {
        return false;
    }
    
    // Administrators always have access
    if (current_user_can('manage_options')) {
        return true;
    }
    
    return current_user_can('coco_ops_view');
}
```

### 2. Added Capabilities to Activation Hook ✅
Now properly adds capabilities during plugin activation:

```php
// Add capabilities to administrator role
$admin_role = get_role('administrator');
if ($admin_role) {
    $admin_role->add_cap('coco_ops_view');
    $admin_role->add_cap('coco_ops_admin');
}

// Add view capability to editor role
$editor_role = get_role('editor');
if ($editor_role) {
    $editor_role->add_cap('coco_ops_view');
}
```

### 3. Added Capability Removal to Deactivation Hook ✅
Properly cleans up capabilities when plugin is deactivated.

## How to Fix Your Installation

### Option 1: Deactivate and Reactivate Plugin (Recommended)
1. **Go to Plugins** in WordPress admin
2. **Deactivate** "Coco Ops Forecast"
3. **Activate** "Coco Ops Forecast" again
4. This will run the activation hook and add the capabilities

### Option 2: Use WP-CLI (If Available)
```bash
wp plugin deactivate coco-ops-forecast
wp plugin activate coco-ops-forecast
```

### Option 3: Manual Capability Addition (Advanced)
If you can't deactivate the plugin, add this code temporarily to your theme's `functions.php`:

```php
// Run this once, then remove
add_action('init', function() {
    $admin_role = get_role('administrator');
    if ($admin_role) {
        $admin_role->add_cap('coco_ops_view');
        $admin_role->add_cap('coco_ops_admin');
    }
});
```

Visit any page on your site, then remove the code.

## After Fixing

### Test REST API Access
Visit these URLs in your browser (while logged in as admin):
- `/wp-json/coco-ops/v1/events` - Should return your events
- `/wp-json/coco-ops/v1/venues` - Should return your venues
- `/wp-json/coco-ops/v1/settings` - Should return settings

### Expected Results
- ✅ No more 401 errors
- ✅ API returns JSON data
- ✅ Events page shows events
- ✅ Forecast dashboard shows event selector
- ✅ Venue dropdown works

## Why This Happened

The original plugin code had an `add_capabilities()` method but it wasn't being called properly during activation. The new code:
1. Adds capabilities directly in the activation hook
2. Has a fallback check for `manage_options` (administrator capability)
3. Ensures administrators always have access

## Verification

After deactivating and reactivating, check:

### Browser Console (F12)
Should see successful API calls:
```
GET /wp-json/coco-ops/v1/events → 200 OK
GET /wp-json/coco-ops/v1/venues → 200 OK
```

### Events Page
- Should load and display events
- No more 401 errors in console

### API Response
Should return data, not errors:
```json
[
  {
    "id": 123,
    "title": "Event Name",
    "venue_name": "Haus Stratford",
    ...
  }
]
```

## Prevention

The updated code now:
- ✅ Adds capabilities during activation
- ✅ Has administrator fallback
- ✅ Removes capabilities during deactivation
- ✅ Properly flushes rewrite rules

This ensures permissions work correctly from the start! 🎯
